package com.kontakt.sdk.android.ble.configuration.scan;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.filter.eddystone.EddystoneFilter;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.Collection;
import java.util.List;

final class ScanContextValidator {

  private ScanContextValidator() {
  }

  private static final int MAX_FILTERS_SIZE = 10;
  private static final int REGIONS_LIMIT = 20;

  static void validate(final ActivityCheckConfiguration activityCheckConfiguration) {
    if (activityCheckConfiguration == ActivityCheckConfiguration.MINIMAL || activityCheckConfiguration == ActivityCheckConfiguration.DISABLED) {
      return;
    }

    long inactivityTimeout = activityCheckConfiguration.getInactivityTimeout();
    long minimalInactivityTimeout = ActivityCheckConfiguration.MINIMAL.getInactivityTimeout();
    SDKPreconditions.checkArgument(inactivityTimeout >= minimalInactivityTimeout, "The inactivity timeout must be no shorter than 3 seconds");

    long checkPeriod = activityCheckConfiguration.getCheckPeriod();
    long minimalCheckPeriod = ActivityCheckConfiguration.MINIMAL.getCheckPeriod();
    SDKPreconditions.checkArgument(checkPeriod >= minimalCheckPeriod, "The check period must be no shorter than 1 second");
  }

  static void validate(final ForceScanConfiguration forceScanConfiguration) {
    if (forceScanConfiguration == ForceScanConfiguration.DISABLED) {
      return;
    }
    final ForceScanConfiguration minimalConfig = ForceScanConfiguration.MINIMAL;

    final long minimalActivePeriod = minimalConfig.getForceScanActivePeriod();
    if (forceScanConfiguration.getForceScanActivePeriod() < minimalActivePeriod) {
      throw new IllegalArgumentException(String.format("The Force Scan Active period must not be shorter than %d milliseconds", minimalActivePeriod));
    }

    final long minimalPassivePeriod = minimalConfig.getForceScanPassivePeriod();
    if (forceScanConfiguration.getForceScanPassivePeriod() < minimalPassivePeriod) {
      throw new IllegalArgumentException(String.format("The Force Scan Passive period must not be shorter than %d", minimalPassivePeriod));
    }
  }

  static void validate(final ScanPeriod scanPeriod) {

    final long activePeriod = scanPeriod.getActivePeriod();
    SDKPreconditions.checkArgument(activePeriod >= ScanPeriod.MINIMAL_ACTIVE_SCAN_PERIOD,
        String.format("Active scan period should last no shorter than %d seconds", ScanPeriod.MINIMAL_ACTIVE_SCAN_PERIOD / 1000));

    final long passiveSubPeriod = scanPeriod.getPassivePeriod();

    SDKPreconditions.checkArgument((passiveSubPeriod == 0L) || (passiveSubPeriod >= ScanPeriod.MINIMAL_PASSIVE_SCAN_PERIOD),
        "Unsupported ScanPeriod's passive period");
  }

  static void validate(final ActivityCheckConfiguration activityCheckConfiguration, ScanPeriod scanPeriod) {
    if (scanPeriod != ScanPeriod.RANGING) {
      SDKPreconditions.checkArgument(activityCheckConfiguration.getCheckPeriod() < scanPeriod.getActivePeriod(),
          "Activity check period must be longer than active scan period");
    }
  }

  static void validateIBeaconRegionsCount(final Collection<IBeaconRegion> iBeaconRegions) {
    SDKPreconditions.checkNotNull(iBeaconRegions, "Regions are null");
    SDKPreconditions.checkArgument(iBeaconRegions.size() <= REGIONS_LIMIT, "You can range beacons within max 20 regions.");
  }

  static void validateIBeaconFiltersCount(List<IBeaconFilter> filters) {
    SDKPreconditions.checkNotNull(filters, "IBeacon filters are null");
    SDKPreconditions.checkState(filters.size() <= MAX_FILTERS_SIZE, "Too many iBeacon filters specified. The limit is " + MAX_FILTERS_SIZE);
  }

  static void validateEddystoneFiltersCount(List<EddystoneFilter> eddystoneFilterList) {
    SDKPreconditions.checkNotNull(eddystoneFilterList, "Eddystone filters are null");
    SDKPreconditions.checkState(eddystoneFilterList.size() <= MAX_FILTERS_SIZE,
        "Too many eddystone filters specified. The limit is " + MAX_FILTERS_SIZE);
  }

  static void validateNamespacesCount(Collection<IEddystoneNamespace> eddystoneNamespaceSet) {
    SDKPreconditions.checkNotNull(eddystoneNamespaceSet, "Eddystone namespaces are null");
    SDKPreconditions.checkArgument(eddystoneNamespaceSet.size() <= REGIONS_LIMIT, "You can range eddystones within max 20 namespaces.");
  }
}