package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;

class SecureProfileResolveRequest {
  private final ISecureProfile fakeDevice;
  private final Integer sourceProximityManagerId;
  private final long timestamp;
  private ResolverType resolvedBy = ResolverType.NONE;
  private ResolveRequestStatus status;

  public SecureProfileResolveRequest(Integer sourceProximityManagerId, ISecureProfile secureProfile, long timestamp) {
    this.sourceProximityManagerId = sourceProximityManagerId;
    this.fakeDevice = secureProfile;
    this.timestamp = timestamp;
  }

  static SecureProfileResolveRequest of(int sourceProximityManagerId, BluetoothDeviceEvent bluetoothDeviceEvent) {
    SecureProfileEvent event = (SecureProfileEvent) bluetoothDeviceEvent;
    EventType eventType = event.getEventType();
    SDKPreconditions.checkArgument(EventType.DEVICE_DISCOVERED == eventType,
        "Cannot create resolve request for other event type than DEVICE_DISCOVERED!");
    List<ISecureProfile> discoveredDeviceList = event.getSecureProfileDevices();
    SDKPreconditions.checkState(discoveredDeviceList.size() == 1, "Resolve request must be created for each one discovered device!");

    ISecureProfile secureProfile = discoveredDeviceList.get(0);
    long timestamp = bluetoothDeviceEvent.getTimestamp();

    return new SecureProfileResolveRequest(sourceProximityManagerId, secureProfile, timestamp);
  }

  public ISecureProfile getFakeDevice() {
    return fakeDevice;
  }

  public Integer getSourceProximityManagerId() {
    return sourceProximityManagerId;
  }

  public long getTimestamp() {
    return timestamp;
  }

  public ResolverType getResolvedBy() {
    return resolvedBy;
  }

  public void setResolvedBy(ResolverType resolvedBy) {
    this.resolvedBy = resolvedBy;
  }

  public void setStatus(ResolveRequestStatus status) {
    this.status = status;
  }

  public ResolveRequestStatus getStatus() {
    return status;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof SecureProfileResolveRequest)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    SecureProfileResolveRequest resolveRequest = (SecureProfileResolveRequest) o;
    return SDKEqualsBuilder.start()
        .equals(fakeDevice, resolveRequest.fakeDevice)
        .equals(sourceProximityManagerId, resolveRequest.sourceProximityManagerId)
        .equals(timestamp, resolveRequest.timestamp)
        .equals(resolvedBy, resolveRequest.resolvedBy)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(fakeDevice).append(sourceProximityManagerId).append(timestamp).append(resolvedBy).build();
  }
}
