package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.paginated.Devices;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.SecureProfileFutureUID;
import com.kontakt.sdk.android.common.model.SecureProfileUid;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

class SecureProfileApiResolveStrategy implements SecureProfileResolveStrategy {

  private final IKontaktCloud kontaktCloud;

  SecureProfileApiResolveStrategy(IKontaktCloud kontaktCloud) {
    this.kontaktCloud = kontaktCloud;
  }

  @Override
  public List<SecureProfileFutureUID> resolve(final Map<SecureProfileUid, SecureProfileResolveRequest> requests) throws Exception {
    final List<SecureProfileUid> toResolve = secureProfilesToResolve(requests);
    if (toResolve.isEmpty()) {
      return Collections.emptyList();
    }

    final List<SecureProfileFutureUID> futureIds = resolveSecureProfiles(toResolve);
    updateRequests(requests, futureIds);
    return futureIds;
  }

  private List<SecureProfileUid> secureProfilesToResolve(final Map<SecureProfileUid, SecureProfileResolveRequest> requests) {
    final List<SecureProfileUid> toResolve = new ArrayList<>();
    for (Map.Entry<SecureProfileUid, SecureProfileResolveRequest> entry : requests.entrySet()) {
      final SecureProfileResolveRequest request = entry.getValue();
      if (ResolveRequestStatus.RESOLVED == request.getStatus()) {
        continue;
      }
      toResolve.add(entry.getKey());
    }
    return toResolve;
  }

  private List<SecureProfileFutureUID> resolveSecureProfiles(final List<SecureProfileUid> uids) throws IOException, KontaktCloudException {
    final Devices cloudResult = kontaktCloud.devices().eddystones().withSecureProfileIds(uids).execute();
    final List<Device> devices = cloudResult.getContent();

    final List<SecureProfileFutureUID> futureIds = new ArrayList<>();
    for (Device device : devices) {
      SecureProfileFutureUID secureProfileFutureUID = device.extractSecureProfileFutureUID();
      if (secureProfileFutureUID != null) {
        futureIds.add(secureProfileFutureUID);
      }
    }
    return futureIds;
  }

  private void updateRequests(Map<SecureProfileUid, SecureProfileResolveRequest> requests, List<SecureProfileFutureUID> futureIds) {
    for (SecureProfileFutureUID id : futureIds) {
      updateRequest(requests, id);
    }
  }

  private void updateRequest(Map<SecureProfileUid, SecureProfileResolveRequest> requests, SecureProfileFutureUID id) {
    SecureProfileResolveRequest request = requests.get(id.getQueriedBy());
    request.setResolvedBy(ResolverType.API);
    if (ResolveRequestStatus.IGNORED == request.getStatus()) {
      return;
    }
    request.setStatus(ResolveRequestStatus.RESOLVED);
  }
}
