package com.kontakt.sdk.android.ble.cache;

import android.content.Context;
import android.util.SparseArray;
import android.util.TimingLogger;
import com.kontakt.sdk.android.ble.configuration.InternalProximityManagerConfiguration;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.manager.listeners.KontaktProximityListener;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import com.kontakt.sdk.android.common.model.IBeaconId;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.model.SecureProfileUid;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SecureProfileUtils;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class FutureShufflesCache {

  private static final String PHANTOM_LABEL = "-PHANTOM-";
  private static final String TAG = "FSCache";
  private static final int DEFAULT_RESOLVER_POOL_SIZE = 2;
  private static final int INITIAL_IBEACON_RESOLVER_DELAY = 1;
  private static final int INITIAL_EDDYSTONE_RESOLVER_DELAY = 2;
  private static final int INITIAL_SECURE_PROFILE_RESOLVER_DELAY = 3;
  public static final ResolvedId PHANTOM_ENTRY = ResolvedId.create(PHANTOM_LABEL, PHANTOM_LABEL, null);

  private final Map<String, ResolvedId> internalCache = new ConcurrentHashMap<>();
  private final Set<String> ignored = new ConcurrentSkipListSet<>();
  private final IBeaconIdResolver iBeaconIdResolver;
  private final EddystoneUIDResolver eddystoneUIDResolver;
  private final SecureProfileResolver secureProfileResolver;
  private final SparseArray<List<KontaktProximityListener>> listenerRegister = new SparseArray<>();
  private final Context context;
  private final InternalProximityManagerConfiguration configuration;
  CacheState state = CacheState.INITIALIZING;
  private ScheduledExecutorService executorService;

  public FutureShufflesCache(Context context, IKontaktCloud kontaktCloud, InternalProximityManagerConfiguration configuration) {
    this.context = context;
    this.configuration = configuration;
    this.iBeaconIdResolver = new IBeaconIdResolver(this, kontaktCloud);
    this.eddystoneUIDResolver = new EddystoneUIDResolver(this, kontaktCloud);
    this.secureProfileResolver = new SecureProfileResolver(this, kontaktCloud);
    new DeserializerThread("cache-deserializer-thread").start();
  }

  public ResolvedId get(RemoteBluetoothDevice origin) {
    switch (origin.getProfile()) {
      case IBEACON:
        IBeaconDevice iBeaconDevice = (IBeaconDevice) origin;
        String fakeId = IBeaconId.fromDevice(iBeaconDevice).toString();
        return get(fakeId);
      case EDDYSTONE:
        IEddystoneDevice eddystoneDevice = (IEddystoneDevice) origin;
        String fakeUID = EddystoneUid.fromDevice(eddystoneDevice).toString();
        return get(fakeUID);
      case KONTAKT_SECURE:
        ISecureProfile secureProfile = SecureProfileUtils.fromRemoteBluetoothDevice(origin);
        String fakeProfileUID = SecureProfileUid.fromDevice(secureProfile).toString();
        return get(fakeProfileUID);
      default:
        throw new IllegalArgumentException("Unsupported device profile: " + origin.getProfile());
    }
  }

  public ResolvedId get(String deviceId) {
    if (internalCache.containsKey(deviceId)) {
      return internalCache.get(deviceId);
    } else if (ignored.contains(deviceId)) {
      return PHANTOM_ENTRY;
    } else {
      return null;
    }
  }

  public void registerProximityManager(int shuffledProximityManagerId) {
    List<KontaktProximityListener> listeners = listenerRegister.get(shuffledProximityManagerId);
    if (listeners != null) {
      return;
    }
    listenerRegister.put(shuffledProximityManagerId, new LinkedList<KontaktProximityListener>());
  }

  public void unregisterProximityManager(int shuffledProximityManagerId) {
    List<KontaktProximityListener> listeners = listenerRegister.get(shuffledProximityManagerId);
    if (listeners == null) {
      return;
    }
    listenerRegister.remove(shuffledProximityManagerId);
  }

  public void addProximityListener(KontaktProximityListener kontaktProximityListener) {
    Integer proximityManagerId = kontaktProximityListener.getParentProximityManagerId();
    List<KontaktProximityListener> listeners = listenerRegister.get(proximityManagerId);
    if (listeners == null) {
      throw new IllegalStateException("Kontakt proximity manager is not registered!");
    }
    createResolverRunners();
    listeners.add(kontaktProximityListener);
  }

  public void removeProximityListener(KontaktProximityListener kontaktProximityListener) {
    Integer proximityManagerId = kontaktProximityListener.getParentProximityManagerId();
    List<KontaktProximityListener> listeners = listenerRegister.get(proximityManagerId);
    if (listeners == null) {
      return;
    }
    listeners.remove(kontaktProximityListener);
    if (listeners.isEmpty()) {
      finishResolveRunners();
    }
  }

  public void addResolveRequest(int proximityManagerId, BluetoothDeviceEvent event) {
    switch (event.getDeviceProfile()) {
      case IBEACON:
        IBeaconResolveRequest iBeaconRequest = IBeaconResolveRequest.of(proximityManagerId, event);
        iBeaconIdResolver.addResolveRequest(iBeaconRequest);
        break;
      case EDDYSTONE:
        EddystoneResolveRequest eddystoneRequest = EddystoneResolveRequest.of(proximityManagerId, event);
        eddystoneUIDResolver.addResolveRequest(eddystoneRequest);
        break;
      case KONTAKT_SECURE:
        SecureProfileResolveRequest secureProfileRequest = SecureProfileResolveRequest.of(proximityManagerId, event);
        secureProfileResolver.addResolveRequest(secureProfileRequest);
        break;
      default:
        throw new IllegalArgumentException("Unsupported device profile: " + event.getDeviceProfile());
    }
  }

  void evict(List<String> uniqueIds, DeviceProfile deviceProfile) {
    Iterator<Map.Entry<String, ResolvedId>> cacheIterator = internalCache.entrySet().iterator();
    while (cacheIterator.hasNext()) {
      Map.Entry<String, ResolvedId> entry = cacheIterator.next();
      ResolvedId evictCandidate = entry.getValue();
      if (uniqueIds.contains(evictCandidate.getUniqueId()) && deviceProfile == evictCandidate.getDeviceProfile()) {
        cacheIterator.remove();
      }
    }
  }

  private void finishResolveRunners() {
    if (executorService != null) {
      executorService.shutdown();
      executorService = null;
    }
  }

  private void createResolverRunners() {
    if (executorService == null) {
      executorService = Executors.newScheduledThreadPool(DEFAULT_RESOLVER_POOL_SIZE);
      executorService.scheduleWithFixedDelay(iBeaconIdResolver, INITIAL_IBEACON_RESOLVER_DELAY, configuration.getResolveInterval(), TimeUnit.SECONDS);
      executorService.scheduleWithFixedDelay(eddystoneUIDResolver, INITIAL_EDDYSTONE_RESOLVER_DELAY, configuration.getResolveInterval(),
          TimeUnit.SECONDS);
      executorService.scheduleWithFixedDelay(secureProfileResolver, INITIAL_SECURE_PROFILE_RESOLVER_DELAY, configuration.getResolveInterval(),
          TimeUnit.SECONDS);    }
  }

  public CacheState getState() {
    return state;
  }

  public boolean isInitialized() {
    return CacheState.INITIALIZED == state;
  }

  void populate(String deviceId, ResolvedId resolvedId) {
    if (PHANTOM_ENTRY.equals(resolvedId)) {
      ignored.add(deviceId);
    } else {
      internalCache.put(deviceId, resolvedId);
    }
  }

  void notifyListeners(int proximityManagerId, BluetoothDeviceEvent event) {
    for (KontaktProximityListener listener : listenerRegister.get(proximityManagerId)) {
      listener.onEvent(event);
    }
  }

  synchronized void serialize() {
    try {
      TimingLogger timings = new TimingLogger(TAG, "Serialization");

      FileOutputStream fos = context.openFileOutput(configuration.getCacheFileName(), Context.MODE_PRIVATE);
      BufferedOutputStream bos = new BufferedOutputStream(fos);
      ObjectOutputStream os = new ObjectOutputStream(fos);

      os.writeObject(internalCache);

      os.close();
      bos.close();
      fos.close();

      Logger.d(TAG + " Cached size: " + internalCache.size());

      timings.addSplit("save file");
      timings.dumpToLog();
    } catch (IOException e) {
      Logger.e(TAG + " Error when try to deserialize cache: ", e);
    }
  }

  void deserialize() {
    try {
      TimingLogger timings = new TimingLogger(TAG, "Deserialization");

      FileInputStream fis = context.openFileInput(configuration.getCacheFileName());
      BufferedInputStream bis = new BufferedInputStream(fis);
      ObjectInputStream ois = new ObjectInputStream(bis);

      ConcurrentHashMap<String, ResolvedId> cached = (ConcurrentHashMap<String, ResolvedId>) ois.readObject();

      ois.close();
      bis.close();
      fis.close();

      Logger.d(TAG + " Cached size: " + cached.size());
      internalCache.putAll(cached);

      timings.addSplit("Read file");
      timings.dumpToLog();
    } catch (FileNotFoundException e) {
      Logger.d(TAG + " Cache file not found: " + e.getMessage());
    } catch (IOException | ClassNotFoundException e) {
      Logger.e(TAG + " Error when try to deserialize cache: ", e);
    }
  }

  public void clear() {
    internalCache.clear();
    clearBuffers();
    File cache = new File(context.getFilesDir(), configuration.getCacheFileName());
    if (!cache.exists()) {
      return;
    }
    cache.delete();
    Logger.d(TAG + " Cache file deleted");
  }

  public synchronized void clearBuffers() {
    ignored.clear();
    iBeaconIdResolver.clear();
    eddystoneUIDResolver.clear();
    secureProfileResolver.clear();
    Logger.d(TAG + "Cleared internal buffers");
  }

  public void markIgnored(RemoteBluetoothDevice beacon) {
    switch (beacon.getProfile()) {
      case IBEACON:
        iBeaconIdResolver.markIgnored(beacon);
        break;
      case EDDYSTONE:
        eddystoneUIDResolver.markIgnored(beacon);
        break;
      case KONTAKT_SECURE:
        secureProfileResolver.markIgnored(beacon);
        break;
      default:
        throw new IllegalArgumentException("Unsupported device profile: " + beacon.getProfile());
    }
  }

  private enum CacheState {
    INITIALIZING,
    INITIALIZED
  }

  private class DeserializerThread extends Thread {

    DeserializerThread(String name) {
      super(name);
    }

    @Override
    public void run() {
      deserialize();
      FutureShufflesCache.this.state = CacheState.INITIALIZED;
    }
  }
}
