/*
 * Copyright (C) 2007 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.kontakt.sdk.android.common.util;

import java.util.Collection;

/**
 * Preconditions provide utility validation methods. For more information
 *
 * @see <a href="http://code.google.com/p/guava-libraries/">Guava Libraries project.</a>
 */
public final class SDKPreconditions {
    private SDKPreconditions() {
    }

    /**
     * Ensures the truth of an expression involving one or more parameters to the calling method.
     *
     * @param expression the expression
     */
    public static void checkArgument(boolean expression) {
        if (!expression) {
            throw new IllegalArgumentException();
        }
    }

    /**
     * Ensures the truth of an expression involving one or more parameters to the calling method.
     *
     * @param expression   the expression
     * @param errorMessage the error message
     */
    public static void checkArgument(boolean expression, Object errorMessage) {
        if (!expression) {
            throw new IllegalArgumentException(String.valueOf(errorMessage));
        }
    }

    /**
     * Ensures the truth of an expression involving one or more parameters to the calling method.
     *
     * @param <E>        the type parameter
     * @param expression the expression
     * @param exception  the exception
     * @throws E the e
     */
    public static <E extends Exception> void checkArgument(boolean expression, E exception) throws E {
        if (!expression) {
            throw exception;
        }
    }

    /**
     * Ensures the truth of an expression involving one or more parameters to the calling method.
     *
     * @param expression           the expression
     * @param errorMessageTemplate the error message template
     * @param errorMessageArgs     the error message args
     */
    public static void checkArgument(boolean expression,
                                     String errorMessageTemplate,
                                     Object... errorMessageArgs) {
        if (!expression) {
            throw new IllegalArgumentException(format(errorMessageTemplate, errorMessageArgs));
        }
    }

    /**
     * Ensures the truth of an expression involving the state of the calling instance,
     * but not involving any parameters to the calling method.
     *
     * @param expression the expression
     */
    public static void checkState(boolean expression) {
        if (!expression) {
            throw new IllegalStateException();
        }
    }

    /**
     * Ensures the truth of an expression involving the state of the calling instance,
     * but not involving any parameters to the calling method.
     *
     * @param <E>        the type parameter
     * @param expression the expression
     * @param exception  the exception
     * @throws E the e
     */
    public static <E extends Exception> void checkState(boolean expression, E exception) throws E {
        if (!expression) {
            throw exception;
        }
    }

    /**
     * Ensures the truth of an expression involving the state of the calling instance,
     * but not involving any parameters to the calling method.
     *
     * @param expression   the expression
     * @param errorMessage the error message
     */
    public static void checkState(boolean expression, Object errorMessage) {
        if (!expression) {
            throw new IllegalStateException(String.valueOf(errorMessage));
        }
    }

    /**
     * Ensures the truth of an expression involving the state of the calling instance,
     * but not involving any parameters to the calling method.
     *
     * @param expression           the expression
     * @param errorMessageTemplate the error message template
     * @param errorMessageArgs     the error message args
     */
    public static void checkState(boolean expression, String errorMessageTemplate, Object... errorMessageArgs) {
        if (!expression) {
            throw new IllegalStateException(format(errorMessageTemplate, errorMessageArgs));
        }
    }

    /**
     * Ensures that an object reference passed as a parameter to the calling method is not null.
     *
     * @param <T>       the type parameter
     * @param reference the reference
     * @return the t
     */
    public static <T> T checkNotNull(T reference) {
        if (reference == null) {
            throw new NullPointerException();
        }
        return reference;
    }

    /**
     * Ensures that an object reference passed as a parameter to the calling method is not null.
     *
     * @param <T>          the type parameter
     * @param reference    the reference
     * @param errorMessage the error message
     * @return the t
     */
    public static <T> T checkNotNull(T reference, Object errorMessage) {
        if (reference == null) {
            throw new NullPointerException(String.valueOf(errorMessage));
        }
        return reference;
    }

    /**
     * Ensures that an object reference passed as a parameter to the calling method is not null.
     *
     * @param <T>                  the type parameter
     * @param reference            the reference
     * @param errorMessageTemplate the error message template
     * @param errorMessageArgs     the error message args
     * @return the t
     */
    public static <T> T checkNotNull(T reference,
                                     String errorMessageTemplate,
                                     Object... errorMessageArgs) {
        if (reference == null) {
            // If either of these parameters is null, the right thing happens anyway
            throw new NullPointerException(format(errorMessageTemplate, errorMessageArgs));
        }
        return reference;
    }


    public static <T, K extends RuntimeException> T[] checkNotAllNull(K exception, T... references) {
        if (exception == null) {
            throw new NullPointerException("Exception cannot be null");
        }
        if (references == null) {
            throw new NullPointerException("References cannot be null");
        }
        int length = references.length;
        int foundNullReferences = 0;

        for (T reference : references) {
            if (reference == null) {
                foundNullReferences++;
            }
        }

        if (foundNullReferences == length) {
            throw exception;
        }

        return references;
    }

    /**
     * Ensures that collection is not null or empty.
     *
     * @param <T>        the type parameter
     * @param collection the collection
     * @param message    the message
     */
    public static <T> void checkNotNullOrEmpty(final Collection<T> collection, final String message) {
        if (collection == null || collection.isEmpty()) {
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Ensures that String is not null or empty.
     *
     * @param target  the target
     * @param message the message
     */
    public static void checkNotNullOrEmpty(final String target, final String message) {
        if (target == null || target.isEmpty()) {
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Ensures that index specifies a valid element in an array, list or string of size size.
     *
     * @param index the index
     * @param size  the size
     * @return the int
     */
    public static int checkElementIndex(int index, int size) {
        return checkElementIndex(index, size, "index");
    }

    /**
     * Ensures that index specifies a valid element in an array, list or string of size size.
     *
     * @param index the index
     * @param size  the size
     * @param desc  the desc
     * @return the int
     */
    public static int checkElementIndex(
            int index, int size, String desc) {
        // Carefully optimized for execution by hotspot (explanatory comment above)
        if (index < 0 || index >= size) {
            throw new IndexOutOfBoundsException(badElementIndex(index, size, desc));
        }
        return index;
    }

    private static String badElementIndex(int index, int size, String desc) {
        if (index < 0) {
            return format("%s (%s) must not be negative", desc, index);
        } else if (size < 0) {
            throw new IllegalArgumentException("negative size: " + size);
        } else { // index >= size
            return format("%s (%s) must be less than size (%s)", desc, index, size);
        }
    }

    /**
     * Ensures that index specifies a valid position in an array, list or string of size size.
     *
     * @param index the index
     * @param size  the size
     * @return the int
     */
    public static int checkPositionIndex(int index, int size) {
        return checkPositionIndex(index, size, "index");
    }

    /**
     * Ensures that index specifies a valid position in an array, list or string of size size.
     *
     * @param index the index
     * @param size  the size
     * @param desc  the desc
     * @return the int
     */
    public static int checkPositionIndex(int index, int size, String desc) {
        // Carefully optimized for execution by hotspot (explanatory comment above)
        if (index < 0 || index > size) {
            throw new IndexOutOfBoundsException(badPositionIndex(index, size, desc));
        }
        return index;
    }

    private static String badPositionIndex(int index, int size, String desc) {
        if (index < 0) {
            return format("%s (%s) must not be negative", desc, index);
        } else if (size < 0) {
            throw new IllegalArgumentException("negative size: " + size);
        } else { // index > size
            return format("%s (%s) must not be greater than size (%s)", desc, index, size);
        }
    }

    /**
     * Ensures that start and end specify a valid positions in an array,
     * list or string of size size, and are in order.
     *
     * @param start the start
     * @param end   the end
     * @param size  the size
     */
    public static void checkPositionIndexes(int start, int end, int size) {
        // Carefully optimized for execution by hotspot (explanatory comment above)
        if (start < 0 || end < start || end > size) {
            throw new IndexOutOfBoundsException(badPositionIndexes(start, end, size));
        }
    }

    public static void checkAllowedSize(Collection<?> collection, int allowedSize, String message) {
        if (collection.size() > allowedSize) {
            throw new IllegalArgumentException(message);
        }
    }

    private static String badPositionIndexes(int start, int end, int size) {
        if (start < 0 || start > size) {
            return badPositionIndex(start, size, "start index");
        }
        if (end < 0 || end > size) {
            return badPositionIndex(end, size, "end index");
        }
        // end < start
        return format("end index (%s) must not be less than start index (%s)", end, start);
    }

    private static String format(String template, Object... args) {
        template = String.valueOf(template);

        StringBuilder builder = new StringBuilder(template.length() + 16 * args.length);
        int templateStart = 0;
        int i = 0;
        while (i < args.length) {
            int placeholderStart = template.indexOf("%s", templateStart);
            if (placeholderStart == -1) {
                break;
            }
            builder.append(template.substring(templateStart, placeholderStart));
            builder.append(args[i++]);
            templateStart = placeholderStart + 2;
        }
        builder.append(template.substring(templateStart));

        if (i < args.length) {
            builder.append(" [");
            builder.append(args[i++]);
            while (i < args.length) {
                builder.append(", ");
                builder.append(args[i++]);
            }
            builder.append(']');
        }

        return builder.toString();
    }
}
