package com.kontakt.sdk.android.common.util;

import java.util.Collection;
import java.util.Iterator;

public abstract class SDKEqualsBuilder {

    private SDKEqualsBuilder() { }

    public abstract SDKEqualsBuilder equals(int lhs, int rhs);

    public abstract SDKEqualsBuilder equals(boolean lhs, boolean rhs);

    public abstract SDKEqualsBuilder equals(long left, long right);

    public abstract SDKEqualsBuilder equals(float left, float right);

    public abstract SDKEqualsBuilder equals(double left, double right);

    public abstract SDKEqualsBuilder equals(Object left, Object right);

    public abstract SDKEqualsBuilder equals(char left, char right);

    public abstract SDKEqualsBuilder equals(Object[] left, Object[] right);

    public abstract SDKEqualsBuilder equals(Collection left, Collection right);

    public abstract boolean result();

    public static SDKEqualsBuilder start() {
        return ACTIVE;
    }

    private static final SDKEqualsBuilder ACTIVE = new SDKEqualsBuilder() {
        public SDKEqualsBuilder equals(int lhs, int rhs) {
            return lhs == rhs ? ACTIVE : FALSE;
        }

        public SDKEqualsBuilder equals(boolean lhs, boolean rhs) {
            return lhs == rhs ? ACTIVE : FALSE;
        }

        public SDKEqualsBuilder equals(long left, long right) {
            return left == right ? ACTIVE : FALSE;
        }
        public SDKEqualsBuilder equals(float left, float right) {
            return left == right ? ACTIVE : FALSE;
        }
        public SDKEqualsBuilder equals(double left, double right) {
            return left == right ? ACTIVE : FALSE;
        }

        @Override
        public SDKEqualsBuilder equals(char left, char right) {
            return left == right ? ACTIVE : FALSE;
        }

        @Override
        public SDKEqualsBuilder equals(Object left, Object right) {
            if(left == right) {
                return ACTIVE;
            }

            if(left == null || right == null) {
                return FALSE;
            }

            if(left.getClass() != right.getClass()) {
                return FALSE;
            }

            return left.equals(right) ? ACTIVE : FALSE;
        }

        @Override
        public SDKEqualsBuilder equals(Object[] left, Object[] right) {
            if(left == right) {
                return ACTIVE;
            }

            if(left == null || right == null) {
                return FALSE;
            }

            if(left.length != right.length) {
                return FALSE;
            }

            for(int i = 0, size = left.length; i < size; i++) {
                if(equals(left[i], right[i]) == FALSE) {
                    return FALSE;
                }
            }

            return ACTIVE;
        }

        @Override
        public SDKEqualsBuilder equals(Collection left, Collection right) {
            if(left == right) {
                return ACTIVE;
            }

            if(left == null || right == null) {
                return FALSE;
            }

            if(left.size() != right.size()) {
                return FALSE;
            }

            Iterator leftIterator = left.iterator();
            Iterator rightIterator = right.iterator();

            while(leftIterator.hasNext()) {
                Object leftObject = leftIterator.next();
                Object rightObject = rightIterator.next();

                if(! leftObject.equals(rightObject)) {
                    return FALSE;
                }
            }

            return ACTIVE;
        }

        public boolean result() {
            return true;
        }

    };

    private static final SDKEqualsBuilder FALSE = new FalseSDKEqualsBuilder();

    private static final class FalseSDKEqualsBuilder extends SDKEqualsBuilder {

        private FalseSDKEqualsBuilder() { }

        @Override
        public SDKEqualsBuilder equals(int lhs, int rhs) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(boolean lhs, boolean rhs) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(long left, long right) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(float left, float right) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(double left, double right) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(Object left, Object right) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(char left, char right) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(Object[] left, Object[] right) {
            return this;
        }

        @Override
        public SDKEqualsBuilder equals(Collection left, Collection right) {
            return this;
        }

        @Override
        public boolean result() {
            return false;
        }
    }

}
