package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.UUID;

/**
 * An abstract representation of manager. By manager we mean person that uses Kontakt.io SDK.
 * The manager can be the owner of multiple {@link Device} and {@link Venue} elements and may
 * belong to a {@link Company}.
 * <br><br>
 * We can obtain managers via {@link IKontaktCloud}.
 * <br><br>
 * To create new instance of this class, please use {@link Manager.Builder}.
 */
public class Manager implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Manager> CREATOR = new Creator<Manager>() {
    @Override public Manager createFromParcel(Parcel source) {
      return new Manager(source);
    }

    @Override public Manager[] newArray(int size) {
      return new Manager[size];
    }
  };

  /**
   * Enumeration type of manager role.
   */
  public enum Role {

    /**
     * Superuser role.
     */
    SUPERUSER,

    /**
     * Admin role.
     */
    ADMIN,

    /**
     * Operator role.
     */
    OPERATOR
  }

  private final String uniqueId;
  private final UUID id;
  private final UUID supervisorId;
  private final String firstName;
  private final String lastName;
  private final String email;
  private final Role role;
  private final Company company;

  public static Builder builder() {
    return new Builder();
  }

  private Manager() {
    this(new Builder());
  }

  private Manager(Builder builder) {
    this.uniqueId = builder.uniqueId;
    this.id = builder.id;
    this.supervisorId = builder.supervisorId;
    this.firstName = builder.firstName;
    this.lastName = builder.lastName;
    this.email = builder.email;
    this.role = builder.role;
    this.company = builder.company;
  }

  protected Manager(Parcel in) {
    this.uniqueId = in.readString();
    this.id = (UUID) in.readSerializable();
    this.supervisorId = (UUID) in.readSerializable();
    this.firstName = in.readString();
    this.lastName = in.readString();
    this.email = in.readString();
    int tmpRole = in.readInt();
    this.role = tmpRole == -1 ? null : Role.values()[tmpRole];
    this.company = in.readParcelable(Company.class.getClassLoader());
  }

  @Override
  public boolean equals(Object object) {
    if (object == this) {
      return true;
    }
    if (object == null || !(object instanceof Manager)) {
      return false;
    }
    final Manager manager = (Manager) object;

    return SDKEqualsBuilder.start()
        .equals(uniqueId, manager.uniqueId)
        .equals(id, manager.id)
        .equals(supervisorId, manager.supervisorId)
        .equals(firstName, manager.firstName)
        .equals(lastName, manager.lastName)
        .equals(email, manager.email)
        .equals(role, manager.role)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(uniqueId)
        .append(id)
        .append(supervisorId)
        .append(firstName)
        .append(lastName)
        .append(email)
        .append(role)
        .build();
  }

  @Override public int describeContents() {
    return 0;
  }

  @Override public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.uniqueId);
    dest.writeSerializable(this.id);
    dest.writeSerializable(this.supervisorId);
    dest.writeString(this.firstName);
    dest.writeString(this.lastName);
    dest.writeString(this.email);
    dest.writeInt(this.role == null ? -1 : this.role.ordinal());
    dest.writeParcelable(this.company, flags);
  }

  public String getUniqueId() {
    return uniqueId;
  }

  public UUID getId() {
    return id;
  }

  public UUID getSupervisorId() {
    return supervisorId;
  }

  public String getFirstName() {
    return firstName;
  }

  public String getLastName() {
    return lastName;
  }

  public String getEmail() {
    return email;
  }

  public Role getRole() {
    return role;
  }

  public Company getCompany() {
    return company;
  }

  public static class Builder {
    private String uniqueId;
    private UUID id;
    private UUID supervisorId;
    private String firstName;
    private String lastName;
    private String email;
    private Role role;
    private Company company;

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder supervisorId(final UUID supervisorId) {
      this.supervisorId = supervisorId;
      return this;
    }

    public Builder firstName(final String firstName) {
      this.firstName = firstName;
      return this;
    }

    public Builder lastName(final String lastName) {
      this.lastName = lastName;
      return this;
    }

    public Builder email(final String email) {
      this.email = email;
      return this;
    }

    public Builder role(final Role role) {
      this.role = role;
      return this;
    }

    public Builder company(final Company company) {
      this.company = company;
      return this;
    }

    public Manager build() {
      return new Manager(this);
    }
  }
}
