package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.List;

/**
 * Represents API response with resolved iBeacon ID (from shuffled) and set of future shuffles.
 */
public class IBeaconFutureId implements IFutureId<IBeaconId> {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<IBeaconFutureId> CREATOR = new Creator<IBeaconFutureId>() {
    @Override public IBeaconFutureId createFromParcel(Parcel source) {
      return new IBeaconFutureId(source);
    }

    @Override public IBeaconFutureId[] newArray(int size) {
      return new IBeaconFutureId[size];
    }
  };

  private String uniqueId;
  private IBeaconId queriedBy;
  private IBeaconId resolved;
  private List<IBeaconId> futureIds;

  public static Builder builder() {
    return new Builder();
  }

  private IBeaconFutureId(Builder builder) {
    this.uniqueId = builder.uniqueId;
    this.queriedBy = builder.queriedBy;
    this.resolved = builder.resolved;
    this.futureIds = builder.futureIds;
  }

  protected IBeaconFutureId(Parcel in) {
    this.uniqueId = in.readString();
    this.queriedBy = in.readParcelable(IBeaconId.class.getClassLoader());
    this.resolved = in.readParcelable(IBeaconId.class.getClassLoader());
    this.futureIds = in.createTypedArrayList(IBeaconId.CREATOR);
  }

  public IBeaconId getQueriedBy() {
    return queriedBy;
  }

  public IBeaconId getResolved() {
    return resolved;
  }

  public List<IBeaconId> getFutureIds() {
    return futureIds;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  @Override public boolean equals(Object o) {
    if (o == null || !(o instanceof IBeaconFutureId)) {
      return false;
    }
    if (o == this) {
      return true;
    }

    IBeaconFutureId other = (IBeaconFutureId) o;
    return SDKEqualsBuilder.start()
        .equals(resolved, other.resolved)
        .equals(queriedBy, other.queriedBy)
        .equals(futureIds, other.futureIds)
        .equals(uniqueId, other.uniqueId)
        .result();
  }

  @Override public int hashCode() {
    return HashCodeBuilder.init()
        .append(resolved)
        .append(queriedBy)
        .append(futureIds)
        .append(uniqueId)
        .build();
  }

  @Override public int describeContents() {
    return 0;
  }

  @Override public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.uniqueId);
    dest.writeParcelable(this.queriedBy, flags);
    dest.writeParcelable(this.resolved, flags);
    dest.writeTypedList(futureIds);
  }

  public static class Builder {
    private String uniqueId;
    private IBeaconId queriedBy;
    private IBeaconId resolved;
    private List<IBeaconId> futureIds = new ArrayList<>();

    public Builder uniqueId(final String uniqueId) {
      this.uniqueId = uniqueId;
      return this;
    }

    public Builder queriedBy(final IBeaconId queriedBy) {
      this.queriedBy = queriedBy;
      return this;
    }

    public Builder resolved(final IBeaconId resolved) {
      this.resolved = resolved;
      return this;
    }

    public Builder futureIds(final List<IBeaconId> futureIds) {
      SDKPreconditions.checkNotNull(futureIds, "future IDs cannot be null");
      this.futureIds.clear();
      this.futureIds.addAll(futureIds);
      return this;
    }

    public IBeaconFutureId build() {
      return new IBeaconFutureId(this);
    }
  }
}
