package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

/**
 * A container for information content properties provided by {@link Action} instances
 * of {@link Action.Type#CONTENT}.
 * <br><br>
 * To create new instance of this class, please use {@link ActionContent.Builder}.
 */
public class ActionContent implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<ActionContent> CREATOR = new Creator<ActionContent>() {
    @Override public ActionContent createFromParcel(Parcel source) {
      return new ActionContent(source);
    }

    @Override public ActionContent[] newArray(int size) {
      return new ActionContent[size];
    }
  };

  /**
   * Enumeration of category type.
   */
  public enum Category {
    /**
     * Image file - image/png, image/jpeg or image/gif.
     */
    IMAGE,

    /**
     * Audio file - audio/mp4, audio/mpeg, audio/ogg, audio/webm or audio/vorbis.
     */
    AUDIO,

    /**
     * Video file - video/mpeg, video/mp4, video/ogg, video/webm, video/quicktime, video/theora or application/x-matroska.
     */
    VIDEO,

    /**
     * Plain text - text/plain.
     */
    TEXT,

    /**
     * Unknown file.
     */
    UNKNOWN
  }

  private int contentLength;
  private String contentType;
  private Category contentCategory;
  private String content;
  private boolean conversionInProgress;

  public static Builder builder() {
    return new Builder();
  }

  private ActionContent() {
    this(new Builder());
  }

  private ActionContent(Builder builder) {
    this.contentLength = builder.contentLength;
    this.contentType = builder.contentType;
    this.contentCategory = builder.contentCategory;
    this.content = builder.content;
    this.conversionInProgress = builder.conversionInProgress;
  }

  protected ActionContent(Parcel in) {
    this.contentLength = in.readInt();
    this.contentType = in.readString();
    int tmpContentCategory = in.readInt();
    this.contentCategory = tmpContentCategory == -1 ? null : Category.values()[tmpContentCategory];
    this.content = in.readString();
    this.conversionInProgress = in.readByte() != 0;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof ActionContent)) {
      return false;
    }
    if (o == this) {
      return true;
    }

    ActionContent other = (ActionContent) o;
    return SDKEqualsBuilder.start()
        .equals(contentLength, other.contentLength)
        .equals(contentType, other.contentType)
        .equals(contentCategory, other.contentCategory)
        .equals(content, other.content)
        .equals(conversionInProgress, other.conversionInProgress)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(contentLength)
        .append(contentType)
        .append(contentCategory)
        .append(content)
        .append(conversionInProgress)
        .build();
  }

  @Override public int describeContents() {
    return 0;
  }

  @Override public void writeToParcel(Parcel dest, int flags) {
    dest.writeInt(this.contentLength);
    dest.writeString(this.contentType);
    dest.writeInt(this.contentCategory == null ? -1 : this.contentCategory.ordinal());
    dest.writeString(this.content);
    dest.writeByte(conversionInProgress ? (byte) 1 : (byte) 0);
  }

  public int getLength() {
    return contentLength;
  }

  public String getType() {
    return contentType;
  }

  public Category getCategory() {
    return contentCategory;
  }

  public String getContentUrl() {
    return content;
  }

  public boolean isConversionInProgress() {
    return conversionInProgress;
  }

  /**
   * Builder class that is used to build {@link ActionContent} instances from values configured by the setters.
   */
  public static class Builder {
    private int contentLength;
    private String contentType;
    private Category contentCategory;
    private String content;
    private boolean conversionInProgress;

    public Builder length(final int contentLength) {
      this.contentLength = contentLength;
      return this;
    }

    public Builder type(final String contentType) {
      this.contentType = contentType;
      return this;
    }

    public Builder category(final Category contentCategory) {
      this.contentCategory = contentCategory;
      return this;
    }

    public Builder content(final String content) {
      this.content = content;
      return this;
    }

    public Builder conversionInProgress(final Boolean conversionInProgress) {
      if (conversionInProgress == null) {
        this.conversionInProgress = false;
      } else {
        this.conversionInProgress = conversionInProgress;
      }
      return this;
    }

    public ActionContent build() {
      return new ActionContent(this);
    }
  }
}
