package com.kontakt.sdk.android.common;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;
import com.kontakt.sdk.android.common.log.LogLevel;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * Entry point for initialization SDK
 */
public final class KontaktSDK {

    /**
     * The constant DEFAULT_KONTAKT_BEACON_PROXIMITY_UUID. This proximity UUID value
     * is preset as the default for every Beacon belonging to Kontakt.
     */
    public static final UUID DEFAULT_KONTAKT_BEACON_PROXIMITY_UUID = UUID.fromString("f7826da6-4fa2-4e98-8024-bc5b71e0893e");

    public static final String DEFAULT_KONTAKT_NAMESPACE_ID = "f7826da6bc5b71e0893e";

    private static KontaktSDK SINGLETON;

    /**
     * Inits SDK with given api key
     *
     * @param apiKey the api key
     * @return KontaktSDK
     */
    public static synchronized KontaktSDK initialize(final String apiKey) {
        SINGLETON = new KontaktSDK(apiKey);

        return SINGLETON;
    }

    /**
     * Inits SDK from Context by reading api key from AndroidManifest
     * <p>
     * throws {@link IllegalStateException} if package name could not been found
     * <br>
     * throws {@link IllegalArgumentException} if api key has not been found in AndroidManifest
     *
     *
     * @param context the context
     * @return KontaktSDK
     */
    public static synchronized KontaktSDK initialize(final Context context) {
        ApplicationInfo appInfo;
        try {
            appInfo = context.getApplicationContext().getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
        } catch (PackageManager.NameNotFoundException e) {
            throw new IllegalStateException("Could not find application package name");
        }

        final String apiKey = appInfo.metaData.getString("kontakt.io.API_KEY");
        SDKPreconditions.checkArgument(!TextUtils.isEmpty(apiKey), "API key is not present. Please provide the key in the AndroidManifest.xml");

        return initialize(apiKey);
    }

    public static synchronized void reset() {
        SINGLETON = null;
    }

    public static synchronized boolean isInitialized() {
        return SINGLETON != null;
    }

    public static synchronized KontaktSDK getInstance() {
        SDKPreconditions.checkNotNull(SINGLETON, "kontakt.io SDK not initialized. Please, invoke initialize() method first");
        return SINGLETON;
    }

    private final String apiKey;

    private KontaktSDK(final String apiKey) {
        this.apiKey = apiKey;
        Logger.reset();
    }

    public final String getApiKey() {
        return apiKey;
    }

    public final KontaktSDK setDebugLoggingEnabled(final boolean state) {
        Logger.setDebugLoggingEnabled(state);
        return this;
    }

    public final KontaktSDK setLogLevelEnabled(final LogLevel level, final boolean state) {
        Logger.setLogLevelEnabled(level, state);
        return this;
    }

    public final KontaktSDK setCrashlyticsLoggingEnabled(final boolean state) {
        Logger.setCrashlyticsLoggingEnabled(state);
        return this;
    }
}
