package com.kontakt.sdk.android.cloud.api.executor.venues;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.VenuesApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.VenuesService;
import com.kontakt.sdk.android.cloud.response.paginated.Venues;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.model.VenueType;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link VenuesApi}. Use this class if you want to fetch
 * devices through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Venues venues = kontaktCloud.venues().fetch()
 *      .managerIds(managerIDs)
 *      .startIndex(5)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class VenuesRequestExecutor extends PaginatedRequestExecutor<Venues> {

  private final VenuesService venuesService;

  private UUID[] managerIds;
  private VenueType type;

  /**
   * Constructs request executor initialized with corresponding service class.
   * @param venuesService the venues API service.
   */
  public VenuesRequestExecutor(final VenuesService venuesService) {
    this.venuesService = venuesService;
  }

  /**
   * Specifies managers.
   * @param managerIds manager unique identifiers.
   * @return this request executor.
   */
  public VenuesRequestExecutor managerIds(final UUID... managerIds) {
    this.managerIds = SDKPreconditions.checkNotNull(managerIds, "managerIds cannot be null");
    return this;
  }

  /**
   * Specifies managers.
   * @param managerIds manager unique identifiers.
   * @return this request executor.
   */
  public VenuesRequestExecutor managerIds(final List<UUID> managerIds) {
    SDKPreconditions.checkNotNull(managerIds, "managerIds cannot be null");
    final int size = managerIds.size();
    this.managerIds = managerIds.toArray(new UUID[size]);
    return this;
  }

  /**
   * Specifies venue type.
   * @param type venue type.
   * @return this request executor.
   */
  public VenuesRequestExecutor type(final VenueType type) {
    this.type = SDKPreconditions.checkNotNull(type, "type cannot be null");
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public VenuesRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public VenuesRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public VenuesRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public VenuesRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public VenuesRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Venues> prepareCall() {
    final Call<Venues> call;
    if (eTag != null) {
      call = venuesService.getVenues(params(), eTag);
    } else {
      call = venuesService.getVenues(params());
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (managerIds != null) {
      params.put(CloudConstants.Venues.MANAGER_ID_PARAMETER, StringUtils.join(managerIds, ","));
    }
    if (type != null) {
      params.put(CloudConstants.Venues.VENUE_TYPE_PARAMETER, type.name());
    }
    return params;
  }
}
