package com.kontakt.sdk.android.cloud.api.executor.proximities;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ProximitiesApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ProximitiesService;
import com.kontakt.sdk.android.cloud.response.paginated.Proximities;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link ProximitiesApi}. Use this class if you want to fetch
 * secure proximities mapping through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   kontaktCloud.proximities().fetch()
 *      .startIndex(4)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class ProximitiesRequestExecutor extends PaginatedRequestExecutor<Proximities> {

  private final ProximitiesService proximitiesService;
  private UUID mappedBy;

  /**
   * Constructs request executor initialized with corresponding service class.
   * @param proximitiesService the proximities API service.
   */
  public ProximitiesRequestExecutor(final ProximitiesService proximitiesService) {
    this.proximitiesService = proximitiesService;
  }

  /**
   * Specifies proximity to mapping.
   * @param proximity the proximity.
   * @return this request executor.
   */
  public ProximitiesRequestExecutor mappedBy(final UUID proximity) {
    SDKPreconditions.checkNotNull(proximity, "proximity cannot be null");
    this.mappedBy = proximity;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public ProximitiesRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public ProximitiesRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public ProximitiesRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public ProximitiesRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public ProximitiesRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Proximities> prepareCall() {
    final Call<Proximities> call;
    if (eTag != null) {
      call = proximitiesService.getProximities(params(), eTag);
    } else {
      call = proximitiesService.getProximities(params());
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (mappedBy != null) {
      params.put(CloudConstants.Proximities.PROXIMITY_PARAMETER, mappedBy.toString());
    }
    return params;
  }
}
