package com.kontakt.sdk.android.cloud.api.executor.namespaces;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.NamespacesApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.NamespacesService;
import com.kontakt.sdk.android.cloud.response.paginated.Namespaces;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link NamespacesApi}. Use this class if you want to fetch
 * secure namespaces mapping through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   kontaktCloud.namespaces().fetch()
 *      .startIndex(4)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class NamespacesRequestExecutor extends PaginatedRequestExecutor<Namespaces> {

  private final NamespacesService namespacesService;
  private String mappedBy;

  /**
   * Constructs request executor initialized with corresponding service class.
   * @param namespacesService the namespaces API service.
   */
  public NamespacesRequestExecutor(final NamespacesService namespacesService) {
    this.namespacesService = namespacesService;
  }

  /**
   * Specifies namespace to mapping.
   * @param namespace the namespace.
   * @return this request executor.
   */
  public NamespacesRequestExecutor mappedBy(final String namespace) {
    SDKPreconditions.checkNotNull(namespace, "namespace cannot be null");
    this.mappedBy = namespace;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public NamespacesRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public NamespacesRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public NamespacesRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public NamespacesRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public NamespacesRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Namespaces> prepareCall() {
    final Call<Namespaces> call;
    if (eTag != null) {
      call = namespacesService.getNamespaces(params(), eTag);
    } else {
      call = namespacesService.getNamespaces(params());
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (mappedBy != null) {
      params.put(CloudConstants.Namespaces.NAMESPACE_PARAMETER, mappedBy);
    }
    return params;
  }
}
