package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to update
 * devices through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   // define updated device object
 *   kontaktCloud.devices().update(ID)
 *      .with(updatedDevice)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that device must be specified so invocation of {@code with} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class UpdateDeviceRequestExecutor extends RequestExecutor<String> {

  private final DevicesService devicesService;
  private final String uniqueId;
  private final DeviceType deviceType;
  private Device device;
  private boolean forceUpdate;

  /**
   * Constructs request executor initialized with corresponding service class and device unique ID.
   * @param devicesService the devices API service.
   * @param uniqueId device unique identifier.
   */
  public UpdateDeviceRequestExecutor(final DevicesService devicesService, final String uniqueId) {
    this.devicesService = devicesService;
    this.uniqueId = uniqueId;
    this.deviceType = DeviceType.BEACON;
  }

  /**
   * Specifies device data to update. The method invocation is obligatory while using this request executor.
   * @param device updated device data.
   * @return this request executor.
   */
  public UpdateDeviceRequestExecutor with(final Device device) {
    SDKPreconditions.checkNotNull(device, "device cannot be null");
    this.device = device;
    return this;
  }

  /**
   * Ignore this parameter if updating device with firmware 4.0 or higher.<br><br>
   * Normally the update request for devices with firmware lower than 4.0 will be revoked if there is a pending config available.<br>
   * Setting force to true will overcome this restriction, update device and remove pending config.
   *
   * @param force true or false
   * @return this request executor.
   */
  public UpdateDeviceRequestExecutor force(boolean force) {
    this.forceUpdate = force;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<String> prepareCall() {
    return devicesService.updateDevice(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(device != null,
        "cannot update device - specify device data to update");
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, uniqueId);
    params.put(CloudConstants.Devices.DEVICE_TYPE_PARAMETER, deviceType.name());
    if (device.getAlias() != null) {
      params.put(CloudConstants.Devices.ALIAS_PARAMETER, device.getAlias());
    }
    if (device.getFirmware() != null) {
      params.put(CloudConstants.Devices.FIRMWARE_PARAMETER, device.getFirmware());
    }
    if (device.getLatitude() != null) {
      params.put(CloudConstants.Common.LAT_PARAMETER, device.getLatitude());
    }
    if (device.getLongitude() != null) {
      params.put(CloudConstants.Common.LNG_PARAMETER, device.getLongitude());
    }
    final Config config = device.getConfig();
    if (config != null) {
      if (config.getProximity() != null) {
        params.put(CloudConstants.Devices.PROXIMITY_PARAMETER, config.getProximity().toString());
      }
      if (config.getMajor() != -1) {
        params.put(CloudConstants.Devices.MAJOR_PARAMETER, String.valueOf(config.getMajor()));
      }
      if (config.getMinor() != -1) {
        params.put(CloudConstants.Devices.MINOR_PARAMETER, String.valueOf(config.getMinor()));
      }
      if (config.getTxPower() != -1) {
        params.put(CloudConstants.Devices.TX_POWER_PARAMETER, String.valueOf(config.getTxPower()));
      }
      if (config.getInterval() != -1) {
        params.put(CloudConstants.Devices.INTERVAL_PARAMETER, String.valueOf(config.getInterval()));
      }
      if (config.getName() != null) {
        params.put(CloudConstants.Devices.NAME_PARAMETER, config.getName());
      }
      if (config.getNamespace() != null) {
        params.put(CloudConstants.Devices.NAMESPACE_PARAMETER, config.getNamespace());
      }
      if (config.getInstanceId() != null) {
        params.put(CloudConstants.Devices.INSTANCE_ID_PARAMETER, config.getInstanceId());
      }
      if (config.getUrl() != null) {
        params.put(CloudConstants.Devices.URL_PARAMETER, config.getHexUrl());
      }
      if (config.getProfiles() != null && !config.getProfiles().isEmpty()) {
        params.put(
            CloudConstants.Devices.PROFILES_PARAMETER, StringUtils.join(config.getProfiles(), ","));
      }
    }

    if (forceUpdate) {
      params.put(CloudConstants.Devices.FORCE_UPDATE, String.valueOf(true));
    }

    return params;
  }
}