package com.kontakt.sdk.android.cloud.api.executor.configs;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ConfigsApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ConfigsService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.PacketType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link ConfigsApi}. Use this class if you want to create
 * configs through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Config config = new Config.Builder().major(532).minor(111).build();
 *   kontaktCloud.configs().create(config)
 *      .forDevices("AxT7", "GVy6", "9KiJ")
 *      .withType(DeviceType.BEACON)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that devices and device type must be specified so invocations of {@code forDevices}
 * and {@code withType} methods are mandatory. Otherwise an exception will be thrown.
 */
public class CreateConfigRequestExecutor extends RequestExecutor<Config[]> {

  private final ConfigsService configsService;

  private final Config config;
  private String[] uniqueIds;
  private DeviceType type;

  /**
   * Constructs request executor initialized with corresponding service class and new config object.
   *
   * @param configsService the configs API service.
   * @param config new config.
   */
  public CreateConfigRequestExecutor(final ConfigsService configsService, final Config config) {
    this.configsService = configsService;
    this.config = config;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CreateConfigRequestExecutor forDevices(final String... uniqueIds) {
    this.uniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CreateConfigRequestExecutor forDevices(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies device type. The method invocation is obligatory while using this request executor.
   *
   * @param type the device type.
   * @return this request executor.
   */
  public CreateConfigRequestExecutor withType(final DeviceType type) {
    this.type = SDKPreconditions.checkNotNull(type, "type cannot be null");
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Config[] execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Config[]> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Config[]> prepareCall() {
    return configsService.createConfig(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(uniqueIds != null, "specify devices");
    SDKPreconditions.checkState(type != null, "specify device type");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    params.put(CloudConstants.Configs.DEVICE_TYPE_PARAMETER, type.name());
    params.put(CloudConstants.Configs.SHUFFLED_PARAMETER, String.valueOf(config.isShuffled()));

    if (config.getName() != null) {
      params.put(CloudConstants.Configs.NAME_PARAMETER, config.getName());
    }
    if (config.getProximity() != null) {
      params.put(CloudConstants.Configs.PROXIMITY_PARAMETER, config.getProximity().toString());
    }
    if (config.getMajor() != -1) {
      params.put(CloudConstants.Configs.MAJOR_PARAMETER, String.valueOf(config.getMajor()));
    }
    if (config.getMinor() != -1) {
      params.put(CloudConstants.Configs.MINOR_PARAMETER, String.valueOf(config.getMinor()));
    }
    if (config.getNamespace() != null) {
      params.put(CloudConstants.Configs.NAMESPACE_PARAMETER, config.getNamespace());
    }
    if (config.getInstanceId() != null) {
      params.put(CloudConstants.Configs.INSTANCE_ID_PARAMETER, config.getInstanceId());
    }
    if (config.getUrl() != null) {
      params.put(CloudConstants.Configs.URL_PARAMETER, config.getHexUrl());
    }
    if (config.getTxPower() != -1) {
      params.put(CloudConstants.Configs.TX_POWER_PARAMETER, String.valueOf(config.getTxPower()));
    }
    if (config.getInterval() != -1) {
      params.put(CloudConstants.Configs.INTERVAL_PARAMETER, String.valueOf(config.getInterval()));
    }
    if (config.getPassword() != null) {
      params.put(CloudConstants.Configs.PASSWORD_PARAMETER, config.getPassword());
    }

    List<DeviceProfile> profiles = config.getProfiles();
    if (profiles != null && !profiles.isEmpty()) {
      params.put(CloudConstants.Configs.PROFILES_PARAMETER, StringUtils.join(profiles, ","));
    }

    List<PacketType> packets = config.getPackets();
    if (packets != null && !packets.isEmpty()) {
      params.put(CloudConstants.Configs.PACKETS_PARAMETER, StringUtils.join(packets, ","));
    }

    return params;
  }
}
