package com.kontakt.sdk.android.cloud.api.executor.actions;

import android.util.Base64;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActionsApi;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.Action;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;
import retrofit2.Call;

/**
 * Request executor provided by {@link ActionsApi}. Use this class if you want to create
 * content actions through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Action contentAction = kontaktCloud.actions().createContentAction()
 *      .forDevices("AxT7")
 *      .withProximity(Proximity.NEAR)
 *      .withMediaFile(new File(fileName))
 *      .execute();
 *   </code>
 * </pre>
 */
public class CreateContentActionRequestExecutor extends CreateActionRequestExecutor {

  private File file;
  private String encodedFile;
  private boolean isMultipart;

  /**
   * Constructs request executor initialized with corresponding service class.
   * @param actionsService the actions API service.
   */
  public CreateContentActionRequestExecutor(final ActionsService actionsService) {
    super(actionsService);
    this.actionType = Action.Type.CONTENT;
  }

  /**
   * Specifies a media file of newly created action.
   * @param file the media file.
   * @return this request executor.
   * @throws IOException instance of IOException
   */
  public CreateContentActionRequestExecutor withMediaFile(final File file) throws IOException {
    SDKPreconditions.checkNotNull(file, "file is null");
    SDKPreconditions.checkState(file.exists(), "file does not exist");
    final byte[] bytes = ConversionUtils.convert(file);
    this.file = file;
    this.encodedFile = Base64.encodeToString(bytes, Base64.DEFAULT);
    return this;
  }

  /**
   * Indicates if create action with multipart or not.
   * @param isMultipart true or false
   * @return this request executor.
   */
  public CreateContentActionRequestExecutor multipart(final boolean isMultipart) {
    this.isMultipart = isMultipart;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public CreateContentActionRequestExecutor forDevices(final String... uniqueIds) {
    super.forDevices(uniqueIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public CreateContentActionRequestExecutor forDevices(final List<String> uniqueIds) {
    super.forDevices(uniqueIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public CreateContentActionRequestExecutor withProximity(final Proximity proximity) {
    super.withProximity(proximity);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Action> prepareCall() {
    if (isMultipart) {
      final RequestBody uniqueIdParam = RequestBody.create(
          MediaType.parse(CloudConstants.MainHeaders.MULTI_PART_FORM_DATA), StringUtils.join(uniqueIds, ","));
      final RequestBody actionTypeParam = RequestBody.create(
          MediaType.parse(CloudConstants.MainHeaders.MULTI_PART_FORM_DATA), actionType.name());
      final RequestBody proximityParam = RequestBody.create(
          MediaType.parse(CloudConstants.MainHeaders.MULTI_PART_FORM_DATA), proximity.name());
      final RequestBody requestFile = RequestBody.create(
          MediaType.parse(CloudConstants.MainHeaders.MULTI_PART_FORM_DATA), file);
      final MultipartBody.Part fileParam = MultipartBody.Part.createFormData(
          CloudConstants.Actions.FILE_PARAMETER, file.getName(), requestFile);

      return actionsService.createContentActionWithMultipart(uniqueIdParam, actionTypeParam, proximityParam, fileParam);
    }
    return super.prepareCall();
  }

  /**
   * {@inheritDoc}
   */
  @Override protected void checkPreconditions() {
    SDKPreconditions.checkState(uniqueIds != null, "cannot create action - specify devices");
    SDKPreconditions.checkState(actionType != null, "cannot create action - specify action type");
    SDKPreconditions.checkState(proximity != null, "cannot create action - specify proximity");
    SDKPreconditions.checkState(encodedFile != null, "cannot create action - specify media file");
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    params.put(CloudConstants.Actions.FILE_PARAMETER, encodedFile);

    return params;
  }
}
