package com.kontakt.sdk.android.cloud.api.executor.actions;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActionsApi;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.Action;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.Map;

/**
 * Request executor provided by {@link ActionsApi}. Use this class if you want to create
 * browser actions through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Action browserAction = kontaktCloud.actions().createBrowserAction()
 *      .forDevices("AxT7")
 *      .withProximity(Proximity.NEAR)
 *      .withUrl("https://kontakt.io")
 *      .execute();
 *   </code>
 * </pre>
 */
public class CreateBrowserActionRequestExecutor extends CreateActionRequestExecutor {

  private String url;

  /**
   * Constructs request executor initialized with corresponding service class.
   * @param actionsService the actions API service.
   */
  public CreateBrowserActionRequestExecutor(final ActionsService actionsService) {
    super(actionsService);
    this.actionType = Action.Type.BROWSER;
  }

  /**
   * Specifies a URL of newly created action.
   * @param url the URL.
   * @return this request executor.
   */
  public CreateBrowserActionRequestExecutor withUrl(final String url) {
    SDKPreconditions.checkNotNullOrEmpty(url, "Url cannot be null or empty");
    this.url = url;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public CreateBrowserActionRequestExecutor forDevices(final String... uniqueIds) {
    super.forDevices(uniqueIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public CreateBrowserActionRequestExecutor forDevices(final List<String> uniqueIds) {
    super.forDevices(uniqueIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public CreateBrowserActionRequestExecutor withProximity(final Proximity proximity) {
    super.withProximity(proximity);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected void checkPreconditions() {
    SDKPreconditions.checkState(uniqueIds != null, "Cannot create action - specify devices");
    SDKPreconditions.checkState(actionType != null, "Cannot create action - specify action type");
    SDKPreconditions.checkState(proximity != null, "Cannot create action - specify proximity");
    SDKPreconditions.checkState(url != null, "Cannot create action - specify url");
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    params.put(CloudConstants.Actions.URL_PARAMETER, url);

    return params;
  }

}
