package com.kontakt.sdk.android.cloud.api.executor;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.HashMap;
import java.util.Map;

/**
 * Fluent API for building requests associated with paginated data.
 * @param <T> the response type.
 */
public abstract class PaginatedRequestExecutor<T> extends RequestExecutor<T> {

  protected int startIndex = -1;
  protected int maxResult = -1;
  protected String query;
  protected String orderBy;
  protected String order;
  protected String eTag;

  /**
   * Specifies the start index of requested data.
   * @param startIndex the start index.
   * @return this request executor.
   */
  public PaginatedRequestExecutor startIndex(final int startIndex) {
    SDKPreconditions.checkState(startIndex >= 0, "start index cannot be negative");
    this.startIndex = startIndex;
    return this;
  }

  /**
   * Specifies the maximum size of response data collection.
   * @param maxResult the maximum response size value.
   * @return this request executor.
   */
  public PaginatedRequestExecutor maxResult(final int maxResult) {
    SDKPreconditions.checkState(maxResult >= 0, "max result cannot be negative");
    this.maxResult = maxResult;
    return this;
  }

  /**
   * Specifies the filter for requested data.
   * @param query the query filter.
   * @return this request executor.
   */
  public PaginatedRequestExecutor filter(final String query) {
    this.query = SDKPreconditions.checkNotNull(query, "query cannot be null");
    return this;
  }

  /**
   * Specifies the ordering of response data elements.
   * @param orderBy the field type that specifies the order by column.
   * @param order ascending or descending.
   * @return this request executor.
   */
  public PaginatedRequestExecutor orderBy(final OrderBy orderBy, final Order order) {
    this.orderBy = SDKPreconditions.checkNotNull(orderBy, "orderBy cannot be null").value();
    this.order = SDKPreconditions.checkNotNull(order, "order cannot be null").name();
    return this;
  }

  /**
   * Specifies HTTP entity tag purposed for caching REST API calls.
   * @param eTag the entity tag value.
   * @return this request executor.
   */
  public PaginatedRequestExecutor eTag(final String eTag) {
    this.eTag = SDKPreconditions.checkNotNull(eTag, "eTag cannot be null");
    return this;
  }

  /**
   * Composes the request's query parameters in map.
   * @return the map of request parameters that are being used to get desired data from API.
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    if (maxResult >= 0) {
      params.put(CloudConstants.Common.MAX_RESULT_PARAMETER, String.valueOf(maxResult));
    }
    if (startIndex >= 0) {
      params.put(CloudConstants.Common.START_INDEX_PARAMETER, String.valueOf(startIndex));
    }
    if (query != null) {
      params.put(CloudConstants.Common.QUERY_PARAMETER, query);
    }
    if (order != null && orderBy != null) {
      params.put(CloudConstants.Common.ORDER_PARAMETER, order);
      params.put(CloudConstants.Common.ORDER_BY_PARAMETER, orderBy);
    }
    return params;
  }
}
