package com.kontakt.sdk.android.cloud.adapter;

import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.model.Action;
import com.kontakt.sdk.android.common.model.ActionContent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * Converts {@link Action} objects to and from JSON.
 */
public class ActionTypeAdapter extends BaseTypeAdapter<Action> {

  /**
   * {@inheritDoc}
   */
  @Override public void write(JsonWriter out, Action action) throws IOException {
    out = out.beginObject();

    writeUUID(out, CloudConstants.Common.ID_PARAMETER, action.getId());
    writeEnum(out, CloudConstants.Actions.ACTION_TYPE_PARAMETER, action.getType());
    writeEnum(out, CloudConstants.Actions.PROXIMITY_PARAMETER, action.getProximity());
    writeString(out, CloudConstants.Actions.URL_PARAMETER, action.getUrl());
    writeContent(out, action.getContent());
    writeDevices(out, action.getDeviceUniqueIds());

    out.endObject();
  }

  /**
   * {@inheritDoc}
   */
  @Override public Action read(JsonReader in) throws IOException {
    final Action.Builder actionBuilder = new Action.Builder();
    final ActionContent.Builder actionContentBuilder = new ActionContent.Builder();

    in.beginObject();
    while (in.hasNext()) {
      String fieldName = in.nextName();
      if (in.peek() == JsonToken.NULL) {
        in.skipValue();
        continue;
      }
      switch (fieldName) {
        case CloudConstants.Common.ID_PARAMETER:
          final UUID id = UUID.fromString(in.nextString());
          actionBuilder.id(id);
          break;
        case CloudConstants.Actions.ACTION_TYPE_PARAMETER:
          final Action.Type type = Action.Type.valueOf(in.nextString());
          actionBuilder.actionType(type);
          break;
        case CloudConstants.Actions.PROXIMITY_PARAMETER:
          final Proximity proximity = Proximity.valueOf(in.nextString());
          actionBuilder.proximity(proximity);
          break;
        case CloudConstants.Actions.URL_PARAMETER:
          final String url = in.nextString();
          actionBuilder.url(url);
          break;
        case CloudConstants.Actions.DEVICE_UNIQUE_IDS_PARAMETER:
          final List<String> deviceUniqueIds = readDevices(in);
          actionBuilder.devices(deviceUniqueIds);
          break;
        case CloudConstants.Actions.CONTENT_PARAMETER:
          final String content = in.nextString();
          actionContentBuilder.content(content);
          break;
        case CloudConstants.Actions.CONTENT_LENGTH_PARAMETER:
          final int contentLength = in.nextInt();
          actionContentBuilder.length(contentLength);
          break;
        case CloudConstants.Actions.CONTENT_TYPE_PARAMETER:
          final String contentType = in.nextString();
          actionContentBuilder.type(contentType);
          break;
        case CloudConstants.Actions.CONTENT_CATEGORY_PARAMETER:
          final ActionContent.Category contentCategory = ActionContent.Category.valueOf(in.nextString());
          actionContentBuilder.category(contentCategory);
          break;
        case CloudConstants.Actions.CONVERSION_IN_PROGRESS_PARAMETER:
          final boolean conversionInProgress = in.nextBoolean();
          actionContentBuilder.conversionInProgress(conversionInProgress);
          break;
        default:
          in.skipValue();
          break;
      }
    }
    in.endObject();

    actionBuilder.content(actionContentBuilder.build());
    return actionBuilder.build();
  }

  private void writeContent(JsonWriter out, ActionContent content) throws IOException {
    if (content == null) {
      return;
    }
    writeInteger(out, CloudConstants.Actions.CONTENT_LENGTH_PARAMETER, content.getLength());
    writeString(out, CloudConstants.Actions.CONTENT_TYPE_PARAMETER, content.getType());
    writeEnum(out, CloudConstants.Actions.CONTENT_CATEGORY_PARAMETER, content.getCategory());
    writeString(out, CloudConstants.Actions.CONTENT_PARAMETER, content.getContentUrl());
    writeBoolean(out, CloudConstants.Actions.CONVERSION_IN_PROGRESS_PARAMETER, content.isConversionInProgress());
  }

  private void writeDevices(JsonWriter out, List<String> deviceUniqueIds) throws IOException {
    out.name(CloudConstants.Actions.DEVICE_UNIQUE_IDS_PARAMETER);
    if (deviceUniqueIds == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (String uniqueId : deviceUniqueIds) {
      out.value(uniqueId);
    }
    out.endArray();
  }

  private List<String> readDevices(JsonReader in) throws IOException {
    in.beginArray();
    List<String> deviceUniqueIds = new ArrayList<>();
    while (in.hasNext()) {
      deviceUniqueIds.add(in.nextString());
    }
    in.endArray();
    return deviceUniqueIds;
  }
}
