package com.kontakt.sdk.android.cloud;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.kontakt.sdk.android.cloud.adapter.ActionTypeAdapter;
import com.kontakt.sdk.android.cloud.adapter.ConfigTypeAdapter;
import com.kontakt.sdk.android.cloud.adapter.DeviceTypeAdapter;
import com.kontakt.sdk.android.cloud.api.ActionsApi;
import com.kontakt.sdk.android.cloud.api.CommandsApi;
import com.kontakt.sdk.android.cloud.api.ConfigsApi;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.EventsApi;
import com.kontakt.sdk.android.cloud.api.FirmwaresApi;
import com.kontakt.sdk.android.cloud.api.ManagersApi;
import com.kontakt.sdk.android.cloud.api.NamespacesApi;
import com.kontakt.sdk.android.cloud.api.NotificationsApi;
import com.kontakt.sdk.android.cloud.api.PresetsApi;
import com.kontakt.sdk.android.cloud.api.ProximitiesApi;
import com.kontakt.sdk.android.cloud.api.VenuesApi;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.cloud.api.service.CommandsService;
import com.kontakt.sdk.android.cloud.api.service.ConfigsService;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.api.service.EventsService;
import com.kontakt.sdk.android.cloud.api.service.FirmwaresService;
import com.kontakt.sdk.android.cloud.api.service.ManagersService;
import com.kontakt.sdk.android.cloud.api.service.NamespacesService;
import com.kontakt.sdk.android.cloud.api.service.NotificationsService;
import com.kontakt.sdk.android.cloud.api.service.PresetsService;
import com.kontakt.sdk.android.cloud.api.service.ProximitiesService;
import com.kontakt.sdk.android.cloud.api.service.VenuesService;
import com.kontakt.sdk.android.cloud.util.ErrorUtils;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.model.Action;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

/**
 * Kontakt.io Proximity REST API client implementation.
 *
 * To obtain Kontakt Cloud instance please use {@link #newInstance(String)} or
 * {@link #newInstance()} method, for example:
 *
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance("my-api-key");
 *   </code>
 * </pre>
 */
public class KontaktCloud implements IKontaktCloud {

  public static final String API_URL = "https://api.kontakt.io";

  public static final int API_VERSION = 9;

  private final ServicesFactory servicesFactory;

  /**
   * Provides new instance of Kontakt Cloud. Please make sure that {@link KontaktSDK} is
   * already initialized with the proper API key.
   * @return the Kontakt.io Proximity REST API client.
   */
  public static KontaktCloud newInstance() {
    return new KontaktCloud(KontaktSDK.getInstance().getApiKey(), API_URL, API_VERSION);
  }

  /**
   * Provides new instance of Kontakt Cloud initialized with given API key.
   * @param apiKey the user's API key.
   * @return the Kontakt.io Proximity REST API client.
   */
  public static KontaktCloud newInstance(final String apiKey) {
    return new KontaktCloud(apiKey, API_URL, API_VERSION);
  }

  private KontaktCloud(final String apiKey, final String apiUrl, final int apiVersion) {
    SDKPreconditions.checkNotNullOrEmpty(apiKey, "Kontakt Cloud - empty api key.");
    SDKPreconditions.checkNotNullOrEmpty(apiUrl, "Kontakt Cloud - empty api url.");
    final Retrofit retrofit = buildRetrofit(apiUrl, buildHttpClient(apiKey, apiVersion));
    this.servicesFactory = ServicesFactory.newInstance(retrofit);
    ErrorUtils.initialize(retrofit);
  }

  /**
   * {@inheritDoc}
   */
  @Override public ActionsApi actions() {
    return new ActionsApi(servicesFactory.createService(ActionsService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public DevicesApi devices() {
    return new DevicesApi(servicesFactory.createService(DevicesService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public VenuesApi venues() {
    return new VenuesApi(servicesFactory.createService(VenuesService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public ConfigsApi configs() {
    return new ConfigsApi(servicesFactory.createService(ConfigsService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public ManagersApi managers() {
    return new ManagersApi(servicesFactory.createService(ManagersService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public FirmwaresApi firmwares() {
    return new FirmwaresApi(servicesFactory.createService(FirmwaresService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public ProximitiesApi proximities() {
    return new ProximitiesApi(servicesFactory.createService(ProximitiesService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public NamespacesApi namespaces() {
    return new NamespacesApi(servicesFactory.createService(NamespacesService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public CommandsApi commands() {
    return new CommandsApi(servicesFactory.createService(CommandsService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public EventsApi events() {
    return new EventsApi(servicesFactory.createService(EventsService.class));
  }

  /**
   * {@inheritDoc}
   */
  @Override public PresetsApi presets() {
    return new PresetsApi(servicesFactory.createService(PresetsService.class));
  }

  private NotificationsApi notificationsApi() {
    return new NotificationsApi(servicesFactory.createService(NotificationsService.class));
  }

  private Retrofit buildRetrofit(final String apiBaseUrl, final OkHttpClient okHttpClient) {
    GsonBuilder gsonBuilder = new GsonBuilder();
    gsonBuilder.registerTypeAdapter(Device.class, new DeviceTypeAdapter());
    gsonBuilder.registerTypeAdapter(Config.class, new ConfigTypeAdapter());
    gsonBuilder.registerTypeAdapter(Action.class, new ActionTypeAdapter());
    gsonBuilder.setPrettyPrinting();
    Gson gson = gsonBuilder.create();

    Retrofit.Builder retrofitBuilder = new Retrofit.Builder()
        .baseUrl(apiBaseUrl)
        .addConverterFactory(GsonConverterFactory.create(gson))
        .client(okHttpClient);
    return retrofitBuilder.build();
  }

  private OkHttpClient buildHttpClient(final String apiKey, final int apiVersion) {
    OkHttpClient.Builder builder = new OkHttpClient.Builder();
    builder.addInterceptor(new Interceptor() {
      @Override public Response intercept(Interceptor.Chain chain) throws IOException {
        Request original = chain.request();
        Request.Builder requestBuilder = original
            .newBuilder()
            .header(CloudConstants.MainHeaders.ACCEPT, "application/vnd.com.kontakt+json;version=" + apiVersion)
            .header(CloudConstants.MainHeaders.API_KEY, apiKey)
            .method(original.method(), original.body());
        Request request = requestBuilder.build();
        return chain.proceed(request);
      }
    });
    return builder.build();
  }
}
