package com.kontakt.sdk.android.ble.util;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.lang.reflect.Method;

/**
 * Bluetooth utility methods.
 */
public final class BluetoothUtils {

  private static final String TAG = BluetoothUtils.class.getSimpleName();

  private BluetoothUtils() {
  }

  /**
   * Is bluetooth enabled.
   *
   * @return the boolean
   */
  public static boolean isBluetoothEnabled() {
    final BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
    return bluetoothAdapter != null && bluetoothAdapter.isEnabled();
  }

  /**
   * Provides information concerning Bluetooth Low Energy support.
   *
   * @param context the context
   * @return the boolean
   */
  public static boolean isBluetoothLeSupported(final Context context) {
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR2) {
      return false;
    }

    SDKPreconditions.checkNotNull(context, "Context is null");
    return context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE);
  }

  /**
   * Provides bluetooth device for specified MAC address.
   *
   * @param address the address
   * @return the bluetooth device or null if bluetooth is turned off
   */
  public static BluetoothDevice getBluetoothDevice(final String address) {
    final BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();

    if (bluetoothAdapter == null) {
      return null;
    }
    return bluetoothAdapter.getRemoteDevice(address);
  }

  /**
   * Refreshs Gatt server.
   *
   * @param gattServer the gatt server
   * @return the boolean
   */
  public static boolean refreshGattServer(final BluetoothGatt gattServer) {
    if (gattServer == null) {
      Logger.e("BluetoothGatt is null. Cannot refresh");
      return false;
    }
    try {
      final Method method = gattServer.getClass().getMethod("refresh", new Class[0]);
      if (method != null) {
        return ((Boolean) method.invoke(gattServer));
      }
    } catch (Exception e) {
      Logger.e(TAG + "Error when refreshing gatt", e);
    }

    return false;
  }
}
