package com.kontakt.sdk.android.ble.spec;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;

/**
 * Telemetry provides meta data specific for particular Eddystone device.
 */
public class Telemetry implements Parcelable {

  /**
   * The parcelable constant CREATOR.
   */
  public static final Creator<Telemetry> CREATOR = new Creator<Telemetry>() {
    @Override
    public Telemetry createFromParcel(Parcel source) {
      Bundle bundle = source.readBundle();

      return new Telemetry.Builder().setVersion(bundle.getInt(Constants.TLM.VERSION))
          .setPduCount(bundle.getInt(Constants.TLM.PDU_COUNT))
          .setTimeSincePowerUp(bundle.getInt(Constants.TLM.TIME_SINCE_POWER_UP))
          .setBatteryVoltage(bundle.getInt(Constants.TLM.BATTERY_VOLTAGE))
          .setTemperature(bundle.getDouble(Constants.TLM.TEMPERATURE))
          .build();
    }

    @Override
    public Telemetry[] newArray(int size) {
      return new Telemetry[size];
    }
  };

  private final int batteryVoltage;
  private final double temperature;
  private final int pduCount;
  private final int timeSincePowerUp;
  private final int version;
  private final HashCodeBuilder hashCodeBuilder;

  /**
   * Instantiates a new Telemetry.
   *
   * @param builder the builder
   */
  public Telemetry(Builder builder) {
    this.batteryVoltage = builder.batteryVoltage;
    this.temperature = builder.temperature;
    this.pduCount = builder.pduCount;
    this.timeSincePowerUp = builder.timeSincePowerUp;
    this.version = builder.version;
    this.hashCodeBuilder = HashCodeBuilder.init();
  }

  /**
   * Provides battery voltage [mV].
   *
   * @return the battery voltage
   */
  public int getBatteryVoltage() {
    return batteryVoltage;
  }

  /**
   * Provides temperature (Celsius degrees).
   *
   * @return the temperature
   */
  public double getTemperature() {
    return temperature;
  }

  /**
   * Provides pdu count since last reboot.
   *
   * @return the pdu count
   */
  public int getPduCount() {
    return pduCount;
  }

  /**
   * Provides time since power up.
   *
   * @return the time since power up
   */
  public int getTimeSincePowerUp() {
    return timeSincePowerUp;
  }

  /**
   * Provides version.
   *
   * @return the version
   */
  public int getVersion() {
    return version;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || !(o instanceof Telemetry)) {
      return false;
    }

    Telemetry telemetry = (Telemetry) o;

    return batteryVoltage == telemetry.batteryVoltage &&
        pduCount == telemetry.pduCount &&
        temperature == telemetry.temperature &&
        timeSincePowerUp == telemetry.timeSincePowerUp &&
        version == telemetry.version;
  }

  @Override
  public int hashCode() {
    return hashCodeBuilder.append(batteryVoltage).append(temperature).append(pduCount).append(timeSincePowerUp).append(version).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    Bundle bundle = new Bundle();
    bundle.putInt(Constants.TLM.BATTERY_VOLTAGE, batteryVoltage);
    bundle.putDouble(Constants.TLM.TEMPERATURE, temperature);
    bundle.putInt(Constants.TLM.PDU_COUNT, pduCount);
    bundle.putInt(Constants.TLM.TIME_SINCE_POWER_UP, timeSincePowerUp);
    bundle.putInt(Constants.TLM.VERSION, version);

    dest.writeBundle(bundle);
  }

  /**
   * Telemetry Builder.
   */
  public static class Builder {
    private int batteryVoltage;
    private double temperature;
    private int pduCount;
    private int timeSincePowerUp;
    private int version;

    /**
     * Sets battery voltage.
     *
     * @param batteryVoltage the battery voltage
     * @return the battery voltage
     */
    public Builder setBatteryVoltage(int batteryVoltage) {
      this.batteryVoltage = batteryVoltage;
      return this;
    }

    /**
     * Sets temperature.
     *
     * @param temperature the temperature
     * @return the temperature
     */
    public Builder setTemperature(double temperature) {
      this.temperature = temperature;
      return this;
    }

    /**
     * Sets pdu count.
     *
     * @param pduCount the pdu count
     * @return the pdu count
     */
    public Builder setPduCount(int pduCount) {
      this.pduCount = pduCount;
      return this;
    }

    /**
     * Sets time since power up.
     *
     * @param timeSincePowerUp the time since power up
     * @return the time since power up
     */
    public Builder setTimeSincePowerUp(int timeSincePowerUp) {
      this.timeSincePowerUp = timeSincePowerUp;
      return this;
    }

    /**
     * Sets version.
     *
     * @param version the version
     * @return the version
     */
    public Builder setVersion(int version) {
      this.version = version;
      return this;
    }

    /**
     * Build telemetry.
     *
     * @return the telemetry
     */
    public Telemetry build() {
      return new Telemetry(this);
    }
  }
}
