package com.kontakt.sdk.android.ble.service;

import com.kontakt.sdk.android.common.log.Logger;
import java.util.concurrent.ThreadFactory;

class ForceScanScheduler {

  static final ForceScanScheduler DISABLED = new ForceScanScheduler(null);

  private static final String FORCE_SCAN_THREAD_TAG = "force-scan-thread";
  private static final ThreadFactory FORCE_SCAN_THREAD_FACTORY = new ForceScanThreadFactory();

  private final Runnable forceScanRunner;
  private Thread forceScanThread;
  private State state;

  ForceScanScheduler(final Runnable forceScanRunner) {
    this.forceScanRunner = forceScanRunner;
    this.state = State.READY;
  }

  synchronized void start() {
    if (!isEnabled()) {
      Logger.d("ForceScanController not enabled");
      return;
    }

    if (isRunning()) {
      Logger.d("ForceScanController already running");
      return;
    }

    if (isReady()) {
      forceScanThread = FORCE_SCAN_THREAD_FACTORY.newThread(forceScanRunner);
      forceScanThread.start();
      setState(State.RUNNING);
    }
  }

  synchronized void stop() {
    if (!isEnabled()) {
      Logger.d("ForceScanController not enabled");
      return;
    }

    if (isRunning()) {
      forceScanThread.interrupt();
      forceScanThread = null;
      setState(State.READY);
    }
  }

  synchronized void finish() {
    if (!isEnabled()) {
      Logger.d("ForceScanController not enabled");
      return;
    }

    if (!isFinished()) {
      if (forceScanThread != null) {
        forceScanThread.interrupt();
        forceScanThread = null;
      }
      setState(State.FINISHED);
    }
  }

  synchronized boolean isRunning() {
    return state == State.RUNNING;
  }

  synchronized boolean isFinished() {
    return state == State.FINISHED;
  }

  synchronized boolean isReady() {
    return state == State.READY;
  }

  boolean isEnabled() {
    return forceScanRunner != null;
  }

  private void setState(final State state) {
    this.state = state;
  }

  private enum State {
    /**
     * The READY.
     */
    READY,
    /**
     * The RUNNING.
     */
    RUNNING,
    /**
     * The FINISHED.
     */
    FINISHED
  }

  private static class ForceScanThreadFactory implements ThreadFactory, Thread.UncaughtExceptionHandler {

    @Override
    public Thread newThread(Runnable r) {
      final Thread thread = new Thread(r, FORCE_SCAN_THREAD_TAG);
      thread.setUncaughtExceptionHandler(this);

      return thread;
    }

    @Override
    public void uncaughtException(Thread thread, Throwable ex) {
      Logger.d(String.format("Force scan interrupted. Exception swallowed."));
    }
  }
}
