package com.kontakt.sdk.android.ble.rssi;

import com.kontakt.sdk.android.common.util.ConversionUtils;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.WeakHashMap;

/**
 * Limited Mean Rssi Calculator provides rssi calculation based on the
 * arithmetic mean of n last rssi records where the n is the parameter
 * applied during calculator construction.
 */
public final class LimitedMeanRssiCalculator implements RssiCalculator {

  private final Map<Integer, Queue<Integer>> rssiMapLog = new WeakHashMap<>();
  private final int logSize;

  LimitedMeanRssiCalculator(final int logSize) {
    this.logSize = logSize;
  }

  @Override
  public int calculateRssi(final int uniqueKey, int rssi) {
    Queue<Integer> rssiQueue = rssiMapLog.get(uniqueKey);
    if (rssiQueue == null) {
      rssiQueue = new LinkedList<>();
      rssiMapLog.put(uniqueKey, rssiQueue);
    }

    if (rssiQueue.size() >= logSize) {
      rssiQueue.poll();
    }

    rssiQueue.add(rssi);

    return mean(ConversionUtils.toPrimitive(rssiQueue.toArray(new Integer[rssiQueue.size()])));
  }

  @Override
  public void clear() {
    rssiMapLog.clear();
  }

  @Override
  public void clear(int uniqueKey) {
    rssiMapLog.remove(uniqueKey);
  }

  private int mean(int[] values) {
    int sum = 0;
    for (int value : values) {
      sum += value;
    }
    return sum / values.length;
  }
}