package com.kontakt.sdk.android.ble.monitoring;

import com.kontakt.sdk.android.ble.configuration.InternalProximityManagerConfiguration;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Event;
import com.kontakt.sdk.android.common.model.EventPacket;
import com.kontakt.sdk.android.common.model.EventType;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class EventCollector implements IEventCollector {

  private static final String TAG = "EventCollector";

  private final EventSender eventSender;
  private InternalProximityManagerConfiguration configuration;
  private ScheduledExecutorService executorService;

  public EventCollector(IKontaktCloud kontaktCloud, InternalProximityManagerConfiguration configuration) {
    this.eventSender = new EventSender(kontaktCloud);
    this.configuration = configuration;
  }

  public void collect(RemoteBluetoothDevice device) {
    this.eventSender.collect(device);
  }

  @Override
  public void stop() {
    if (executorService == null) {
      return;
    }
    executorService.shutdown();
    executorService = null;
  }

  @Override
  public void clear() {
    this.eventSender.clear();
  }

  @Override
  public void start() {
    if (executorService != null || !configuration.isMonitoringEnabled()) {
      return;
    }
    executorService = Executors.newSingleThreadScheduledExecutor();
    executorService.scheduleWithFixedDelay(eventSender, configuration.getMonitoringSyncInterval(), configuration.getMonitoringSyncInterval(), TimeUnit.SECONDS);
  }

  @Override
  public void updateConfiguration(InternalProximityManagerConfiguration configuration) {
    this.configuration = configuration;
  }

  static class EventSender implements Runnable {

    private static final int DEFAULT_BUFFER_SIZE = 200;
    private final ArrayBlockingQueue<Event> eventsBuffer;
    private final IKontaktCloud kontaktCloud;
    private final ConcurrentSkipListSet<String> ignored = new ConcurrentSkipListSet<>();

    EventSender(final IKontaktCloud kontaktCloud) {
      this.eventsBuffer = new ArrayBlockingQueue<>(DEFAULT_BUFFER_SIZE, true);
      this.kontaktCloud = kontaktCloud;
    }

    @Override
    public void run() {
      final List<Event> events = new ArrayList<>();
      eventsBuffer.drainTo(events);
      if (events.isEmpty()) {
        Logger.d(TAG + " Nothing to send");
        return;
      }

      final EventPacket eventPacket = EventPacket.from(events, System.currentTimeMillis());
      try {
        kontaktCloud.events()
            .collect(eventPacket)
            .execute();
      } catch (IOException | KontaktCloudException e) {
        Throwable cause = e.getCause();
        if (UnknownHostException.class.isInstance(cause) || SocketTimeoutException.class.isInstance(cause)) {
          eventsBuffer.addAll(events);
        } else {
          Logger.e(TAG + " Error occurred when try to send monitoring events", e);
        }
      }
    }

    void collect(final RemoteBluetoothDevice device) {
      final String notificationId = device.getUniqueId() + ":" + device.getBatteryPower();
      if (ignored.contains(notificationId)) {
        return;
      }
      Event event = Event.of(device, EventType.MONITORING);
      try {
        eventsBuffer.add(event);
        ignored.add(notificationId);
      } catch (IllegalStateException e) {
        Logger.e("Event collector queue is full", e);
      }
    }

    public void clear() {
      eventsBuffer.clear();
      ignored.clear();
    }
  }
}
