package com.kontakt.sdk.android.ble.manager.service;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * AbstractServiceConnector provides an abstraction to every Service. The abstraction
 * also validates required permissions specified when creating new ServiceConnector instance.
 */
public abstract class AbstractServiceConnector implements ServiceConnector {

  private final String[] permissions;
  private final String[] permissionsMarshmallow;

  /**
   * The Context instance.
   */
  protected Context context;

  /**
   * Instantiates a new Abstract service connector.
   *
   * @param context the context
   * @param permissions the permissions
   */
  protected AbstractServiceConnector(final Context context, final String[] permissions, final String[] permissionsMarshmallow) {
    SDKPreconditions.checkState(context != null, "Applied context instance is null");
    SDKPreconditions.checkState(permissions != null, "Permissions not specified");
    this.context = context.getApplicationContext();
    this.permissions = permissions;
    this.permissionsMarshmallow = permissionsMarshmallow;
  }

  @Override
  public void disconnect() {
    Logger.d(getClass().getSimpleName() + " disconnected.");
  }

  /**
   * Checks specific permissions under which Service connector works.
   */
  @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1)
  protected void checkPermissions() {
    checkStandardPermissions();
    checkMarshmallowPermissions();
  }

  private void checkStandardPermissions() {
    for (final String permission : permissions)
      if (context.checkCallingOrSelfPermission(permission) != PackageManager.PERMISSION_GRANTED) {
        throw new RuntimeException(String.format("Permission %s is not granted", permission));
      }
  }

  private void checkMarshmallowPermissions() {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
      for (String permission : permissionsMarshmallow) {
        if (context.checkCallingOrSelfPermission(permission) == PackageManager.PERMISSION_GRANTED) {
          return;
        }
      }
      throw new RuntimeException(String.format("Permission ACCESS_COARSE_LOCATION or ACCESS_FINE_LOCATION must be granted for android Marshmallow"));
    }
  }
}
