package com.kontakt.sdk.android.ble.manager.listeners;

import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.device.BeaconDevice;
import com.kontakt.sdk.android.ble.device.EddystoneDevice;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconDeviceEvent;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.monitoring.IEventCollector;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.LinkedList;
import java.util.List;

public class KontaktProximityListener implements InternalProximityListener {

  private static final String TAG = "KPListener";
  private final int parentProximityManagerId;
  private final FutureShufflesCache cache;
  private final IEventCollector eventCollector;
  private InternalProximityListener internalProximityListener;

  public KontaktProximityListener(int parentProximityManagerId, InternalProximityListener internalProximityListener,
      FutureShufflesCache futureShuffleCache, IEventCollector eventCollector) {
    this.parentProximityManagerId = parentProximityManagerId;
    this.internalProximityListener = internalProximityListener;
    this.cache = futureShuffleCache;
    this.eventCollector = eventCollector;
  }

  @Override
  public void onScanStart() {
    internalProximityListener.onScanStart();
  }

  @Override
  public void onScanStop() {
    internalProximityListener.onScanStop();
  }

  public void onScanError(ScanError exception) {
    internalProximityListener.onScanError(exception);
  }

  @Override
  public void onMonitoringCycleStart() {
    internalProximityListener.onMonitoringCycleStart();
  }

  @Override
  public void onMonitoringCycleStop() {
    internalProximityListener.onMonitoringCycleStop();
  }

  @Override
  public void onEvent(BluetoothDeviceEvent event) {
    switch (event.getEventType()) {
      case DEVICE_DISCOVERED:
        handleDiscoverEvent(event);
        break;
      case DEVICES_UPDATE:
        handleUpdateEvent(event);
        break;
      case DEVICE_LOST:
        handleLostEvent(event);
        break;
      default:
        internalProximityListener.onEvent(event);
    }
  }

  private void handleDiscoverEvent(BluetoothDeviceEvent event) {
    List<? extends RemoteBluetoothDevice> originalDevices = event.getDeviceList();

    SDKPreconditions.checkArgument(originalDevices.size() == 1, "Only one device can be discovered at once!");
    List<RemoteBluetoothDevice> resolvedDevices = new LinkedList<>();
    for (RemoteBluetoothDevice origin : originalDevices) {
      if (!origin.isShuffled()) {
        Logger.d(TAG + " Discover not shuffled device");
        eventCollector.collect(origin);
        resolvedDevices.add(origin);
        continue;
      }

      Logger.d(TAG + " Discover shuffled device");
      ResolvedId resolvedId = cache.get(origin);
      if (resolvedId == null) {
        cache.addResolveRequest(parentProximityManagerId, event);
        continue;
      }
      if (FutureShufflesCache.PHANTOM_ENTRY.equals(resolvedId)) {
        continue;
      }
      if (origin.getUniqueId() == null) {
        RemoteBluetoothDevice resolvedDevice = getBluetoothDevice(origin, resolvedId);
        eventCollector.collect(resolvedDevice);
        resolvedDevices.add(resolvedDevice);
      } else {
        // if origin.uniqueId != null && origin.isShuffled
        // then the event was sent from resolvers
        // and can be directly further processed
        eventCollector.collect(origin);
        internalProximityListener.onEvent(event);
      }
    }

    if (resolvedDevices.isEmpty()) {
      return;
    }
    BluetoothDeviceEvent updatedEvent = getBluetoothDeviceEvent(event, resolvedDevices);
    internalProximityListener.onEvent(updatedEvent);
  }

  private void handleUpdateEvent(BluetoothDeviceEvent event) {
    List<? extends RemoteBluetoothDevice> originalDevices = event.getDeviceList();

    List<RemoteBluetoothDevice> resolvedDevices = new LinkedList<>();

    for (RemoteBluetoothDevice origin : originalDevices) {
      if (origin == null) {
        continue;
      }
      if (!origin.isShuffled()) {
        resolvedDevices.add(origin);
        continue;
      }

      ResolvedId resolvedId = cache.get(origin);
      // not resolved yet
      if (resolvedId == null || FutureShufflesCache.PHANTOM_ENTRY.equals(resolvedId)) {
        continue;
      }
      RemoteBluetoothDevice resolvedDevice = getBluetoothDevice(origin, resolvedId);
      resolvedDevices.add(resolvedDevice);
    }
    if (resolvedDevices.isEmpty()) {
      return;
    }
    BluetoothDeviceEvent updatedEvent = getBluetoothDeviceEvent(event, resolvedDevices);
    internalProximityListener.onEvent(updatedEvent);
  }

  private void handleLostEvent(BluetoothDeviceEvent event) {
    List<? extends RemoteBluetoothDevice> originalDevices = event.getDeviceList();

    List<RemoteBluetoothDevice> resolvedDevices = new LinkedList<>();

    for (RemoteBluetoothDevice origin : originalDevices) {
      if (!origin.isShuffled()) {
        resolvedDevices.add(origin);
        continue;
      }

      ResolvedId resolvedId = cache.get(origin);
      // not resolved yet
      if (resolvedId == null) {
        cache.markIgnored(origin);
        continue;
      }
      if (FutureShufflesCache.PHANTOM_ENTRY.equals(resolvedId)) {
        continue;
      }
      RemoteBluetoothDevice resolvedDevice = getBluetoothDevice(origin, resolvedId);
      resolvedDevices.add(resolvedDevice);
    }
    if (resolvedDevices.isEmpty()) {
      return;
    }
    BluetoothDeviceEvent updatedEvent = getBluetoothDeviceEvent(event, resolvedDevices);
    internalProximityListener.onEvent(updatedEvent);
  }

  public int getParentProximityManagerId() {
    return parentProximityManagerId;
  }

  private static RemoteBluetoothDevice getBluetoothDevice(RemoteBluetoothDevice bluetoothDevice, ResolvedId resolvedId) {
    switch (bluetoothDevice.getProfile()) {
      case IBEACON:
        IBeaconDevice iBeaconDevice = (IBeaconDevice) bluetoothDevice;
        return BeaconDevice.of(iBeaconDevice, resolvedId);
      case EDDYSTONE:
        IEddystoneDevice eddystoneDevice = (IEddystoneDevice) bluetoothDevice;
        return new EddystoneDevice.Builder(eddystoneDevice).setResolvedId(resolvedId).build();
      default:
        throw new IllegalArgumentException("Unsupported device profile!");
    }
  }

  private static BluetoothDeviceEvent getBluetoothDeviceEvent(BluetoothDeviceEvent bluetoothDeviceEvent,
      List<? extends RemoteBluetoothDevice> devices) {
    switch (bluetoothDeviceEvent.getDeviceProfile()) {
      case IBEACON:
        IBeaconDeviceEvent iBeaconDeviceEvent = (IBeaconDeviceEvent) bluetoothDeviceEvent;
        List<IBeaconDevice> iBeaconDevices = (List<IBeaconDevice>) devices;
        return IBeaconDeviceEvent.of(iBeaconDeviceEvent, iBeaconDevices);
      case EDDYSTONE:
        EddystoneDeviceEvent eddystoneDeviceEvent = (EddystoneDeviceEvent) bluetoothDeviceEvent;
        List<IEddystoneDevice> eddystoneDevices = (List<IEddystoneDevice>) devices;
        return EddystoneDeviceEvent.of(eddystoneDeviceEvent, eddystoneDevices);
      default:
        throw new IllegalArgumentException("Unsupported device profile!");
    }
  }
}
