package com.kontakt.sdk.android.ble.manager.internal;

import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.util.ScanContextUtils;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

class ShuffledSpacesManager {

  interface OnSpacesResolvedListener {
    void onSpacesResolved(ScanContext scanContext);

    void onError(ScanError error);
  }

  private final List<SpacesResolver> resolvers = new ArrayList<>();
  private OnSpacesResolvedListener onSpacesResolvedListener;
  private ScanContext scanContext;

  public ShuffledSpacesManager(IKontaktCloud cloud) {
    resolvers.add(new SpacesResolver.RegionsResolver(cloud));
    resolvers.add(new SpacesResolver.NamespacesResolver(cloud));
  }

  public void resolve(ScanContext scanContext, OnSpacesResolvedListener onSpacesResolvedListener) {
    this.scanContext = checkNotNull(scanContext);
    this.onSpacesResolvedListener = checkNotNull(onSpacesResolvedListener);
    for (SpacesResolver resolver : resolvers) {
      resolver.resolve(scanContext, resolvingStatusListener);
    }
  }

  public void clearCache() {
    for (SpacesResolver resolver : resolvers) {
      resolver.clearCache();
    }
  }

  public void onDestroy() {
    onSpacesResolvedListener = null;
  }

  private ScanContext createScanContext(Collection<IBeaconRegion> resolvedRegions, Collection<IEddystoneNamespace> resolvedNamespaces) {
    return ScanContextUtils.cloneWithNewSpaces(scanContext, resolvedRegions, resolvedNamespaces);
  }

  private final SpacesResolver.ResolvingStatusListener resolvingStatusListener = new SpacesResolver.ResolvingStatusListener() {
    @Override
    public void onSuccess() {
      if(allResolversAreFinishedSuccessfully()) {
        Collection<IBeaconRegion> regions = resolvers.get(0).getResolvedSpaces();
        Collection<IEddystoneNamespace> namespaces = resolvers.get(1).getResolvedSpaces();
        resetFinishedStatus();
        onSpacesResolvedListener.onSpacesResolved(createScanContext(regions, namespaces));
      }
    }

    @Override
    public void onError(String message) {
      onSpacesResolvedListener.onError(new ScanError(message));
    }
  };

  private void resetFinishedStatus() {
    for (SpacesResolver resolver : resolvers) {
      resolver.resetFinishedStatus();
    }
  }

  private boolean allResolversAreFinishedSuccessfully() {
    for (SpacesResolver resolver : resolvers) {
      if(!resolver.isFinishedSuccessfully()) {
        return false;
      }
    }
    return true;
  }

}
