package com.kontakt.sdk.android.ble.manager.configuration;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.InternalProximityManagerConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.configuration.scan.ScanMode;
import com.kontakt.sdk.android.ble.filter.eddystone.EddystoneFilter;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * This class is responsible for ranging/monitoring configuration.
 */
public class Configuration implements GeneralConfigurator, SpacesConfigurator, FiltersConfigurator {

  private ScanPeriod scanPeriod;
  private ScanMode scanMode;
  private ForceScanConfiguration forceScanConfiguration;
  private ActivityCheckConfiguration activityCheckConfiguration;
  private RssiCalculator rssiCalculator;
  private Collection<EddystoneFrameType> eddystoneFrameTypes;
  private boolean isNonConnectableSupported;
  private Set<DeviceProfile> observedProfiles;

  private String cacheFileName;
  private boolean monitoringEnabled;
  private int monitoringSyncIntervalSeconds;
  private int resolveShuffledInterval;
  private long deviceUpdateCallbackInterval;

  private Collection<IBeaconRegion> iBeaconRegions;
  private Collection<IEddystoneNamespace> eddystoneNamespaces;
  private Collection<IBeaconFilter> iBeaconFilters;
  private Collection<EddystoneFilter> eddystoneFilters;

  public Configuration(ScanContext scanContext, InternalProximityManagerConfiguration proximityManagerConfiguration) {
    checkNotNull(scanContext);
    checkNotNull(proximityManagerConfiguration);

    this.scanPeriod = scanContext.getScanPeriod();
    this.forceScanConfiguration = scanContext.getForceScanConfiguration();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.scanMode = scanContext.getScanMode();
    this.deviceUpdateCallbackInterval = scanContext.getDevicesUpdateCallbackInterval();
    this.rssiCalculator = scanContext.getRssiCalculator();
    this.eddystoneFrameTypes = Collections.unmodifiableCollection(scanContext.getEddystoneFrameTypes());
    this.isNonConnectableSupported = scanContext.isNonConnectableModeSupported();
    this.observedProfiles = scanContext.getObservedProfiles();

    this.cacheFileName = proximityManagerConfiguration.getCacheFileName();
    this.monitoringEnabled = proximityManagerConfiguration.isMonitoringEnabled();
    this.monitoringSyncIntervalSeconds = proximityManagerConfiguration.getMonitoringSyncInterval();
    this.resolveShuffledInterval = proximityManagerConfiguration.getResolveInterval();

    this.iBeaconRegions = Collections.unmodifiableCollection(scanContext.getIBeaconRegions());
    this.eddystoneNamespaces = Collections.unmodifiableCollection(scanContext.getEddystoneNamespaces());
    this.iBeaconFilters = Collections.unmodifiableCollection(scanContext.getIBeaconFilters());
    this.eddystoneFilters = Collections.unmodifiableCollection(scanContext.getEddystoneFilters());
  }

  public ScanContext createScanContext() {
    return new ScanContext.Builder()
        .setScanPeriod(scanPeriod)
        .setScanMode(scanMode)
        .setDevicesUpdateCallbackInterval(deviceUpdateCallbackInterval)
        .setForceScanConfiguration(forceScanConfiguration)
        .setActivityCheckConfiguration(activityCheckConfiguration)
        .setRssiCalculator(rssiCalculator)
        .setSupportNonConnectableMode(isNonConnectableSupported)
        .setObservedProfiles(observedProfiles)
        .setIBeaconRegions(iBeaconRegions)
        .setIBeaconFilters(iBeaconFilters)
        .setEddystoneNamespaces(eddystoneNamespaces)
        .setEddystoneFilters(eddystoneFilters)
        .setRequiredEddystoneFrameTypes(eddystoneFrameTypes)
        .build();
  }

  public InternalProximityManagerConfiguration createKontaktManagerConfiguration() {
    return new InternalProximityManagerConfiguration.Builder()
        .setMonitoringEnabled(monitoringEnabled)
        .setMonitoringSyncInterval(monitoringSyncIntervalSeconds)
        .setCacheFileName(cacheFileName)
        .setResolveInterval(resolveShuffledInterval)
        .build();
  }

  @Override
  public GeneralConfigurator scanPeriod(ScanPeriod scanPeriod) {
    this.scanPeriod = checkNotNull(scanPeriod);
    return this;
  }

  @Override
  public GeneralConfigurator scanMode(ScanMode scanMode) {
    this.scanMode = checkNotNull(scanMode);
    return this;
  }

  @Override
  public GeneralConfigurator forceScanConfiguration(ForceScanConfiguration forceScanConfiguration) {
    this.forceScanConfiguration = checkNotNull(forceScanConfiguration);
    return this;
  }

  @Override
  public GeneralConfigurator activityCheckConfiguration(ActivityCheckConfiguration activityCheckConfiguration) {
    this.activityCheckConfiguration = checkNotNull(activityCheckConfiguration);
    return this;
  }

  @Override
  public GeneralConfigurator cacheFileName(String name) {
    this.cacheFileName = checkNotNull(name);
    return this;
  }

  @Override
  public GeneralConfigurator monitoringEnabled(boolean enabled) {
    this.monitoringEnabled = checkNotNull(enabled);
    return this;
  }

  @Override
  public GeneralConfigurator monitoringSyncInterval(int intervalInSeconds) {
    checkArgument(intervalInSeconds > 0);
    this.monitoringSyncIntervalSeconds = intervalInSeconds;
    return this;
  }

  @Override
  public GeneralConfigurator resolveShuffledInterval(int intervalInSeconds) {
    checkArgument(intervalInSeconds > 0);
    this.resolveShuffledInterval = intervalInSeconds;
    return this;
  }

  @Override
  public GeneralConfigurator deviceUpdateCallbackInterval(long intervalInMillis) {
    checkArgument(intervalInMillis > 0);
    this.deviceUpdateCallbackInterval = intervalInMillis;
    return this;
  }

  @Override
  public GeneralConfigurator rssiCalculator(RssiCalculator rssiCalculator) {
    this.rssiCalculator = checkNotNull(rssiCalculator);
    return this;
  }

  @Override
  public GeneralConfigurator eddystoneFrameTypes(Collection<EddystoneFrameType> eddystoneFrameTypes) {
    checkNotNull(eddystoneFrameTypes);
    this.eddystoneFrameTypes = Collections.unmodifiableCollection(eddystoneFrameTypes);
    return this;
  }

  @Override
  public GeneralConfigurator supportNonConnectableMode(boolean enabled) {
    this.isNonConnectableSupported = enabled;
    return this;
  }

  @Override
  public SpacesConfigurator iBeaconRegion(IBeaconRegion region) {
    checkNotNull(region);
    iBeaconRegions(Collections.singletonList(region));
    return this;
  }

  @Override
  public SpacesConfigurator iBeaconRegions(Collection<IBeaconRegion> regions) {
    checkNotNull(regions);
    this.iBeaconRegions = Collections.unmodifiableCollection(regions);
    return this;
  }

  @Override
  public SpacesConfigurator eddystoneNamespace(IEddystoneNamespace namespace) {
    checkNotNull(namespace);
    eddystoneNamespaces(Collections.singletonList(namespace));
    return this;
  }

  @Override
  public SpacesConfigurator eddystoneNamespaces(Collection<IEddystoneNamespace> namespaces) {
    checkNotNull(namespaces);
    this.eddystoneNamespaces = Collections.unmodifiableCollection(namespaces);
    return this;
  }

  @Override
  public FiltersConfigurator iBeaconFilter(IBeaconFilter filter) {
    checkNotNull(filter);
    iBeaconFilters(Collections.singletonList(filter));
    return this;
  }

  @Override
  public FiltersConfigurator iBeaconFilters(Collection<IBeaconFilter> filters) {
    checkNotNull(filters);
    this.iBeaconFilters = Collections.unmodifiableCollection(filters);
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneFilter(EddystoneFilter filter) {
    checkNotNull(filter);
    eddystoneFilters(Collections.singletonList(filter));
    return this;
  }

  @Override
  public FiltersConfigurator eddystoneFilters(Collection<EddystoneFilter> filters) {
    checkNotNull(filters);
    this.eddystoneFilters = Collections.unmodifiableCollection(filters);
    return this;
  }

  @Override
  public void clearAll() {
    iBeaconFilters = Collections.emptyList();
    eddystoneFilters = Collections.emptyList();
  }

  public void addObservedProfiles(DeviceProfile deviceProfile) {
    observedProfiles.add(deviceProfile);
  }

  public void removeObservedProfiles(DeviceProfile deviceProfile) {
    observedProfiles.remove(deviceProfile);
  }

}
