package com.kontakt.sdk.android.ble.manager;

import android.content.Context;
import com.kontakt.sdk.android.ble.configuration.InternalProximityManagerConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;
import com.kontakt.sdk.android.ble.manager.configuration.Configuration;
import com.kontakt.sdk.android.ble.manager.configuration.FiltersConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.GeneralConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.SpacesConfigurator;
import com.kontakt.sdk.android.ble.manager.internal.EventObserver;
import com.kontakt.sdk.android.ble.manager.internal.InternalProximityManager;
import com.kontakt.sdk.android.ble.manager.listeners.EddystoneListener;
import com.kontakt.sdk.android.ble.manager.listeners.IBeaconListener;
import com.kontakt.sdk.android.ble.manager.listeners.ScanStatusListener;
import com.kontakt.sdk.android.ble.manager.listeners.SpaceListener;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.DeviceProfile;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public class ProximityManager implements ProximityManagerContract {

  private final Configuration configuration;

  private IBeaconListener iBeaconListener;
  private EddystoneListener eddystoneListener;
  private ScanStatusListener scanStatusListener;
  private SpaceListener spaceListener;
  private InternalProximityManager internalProximityManager;
  private ScanContext scanContext = ScanContext.DEFAULT;
  private InternalProximityManagerConfiguration internalProximityManagerConfiguration = InternalProximityManagerConfiguration.DEFAULT;

  public ProximityManager(Context context) {
    checkNotNull(context, "Context can't be null");
    this.configuration = new Configuration(scanContext, internalProximityManagerConfiguration);
    this.internalProximityManager = new InternalProximityManager(context.getApplicationContext());
  }

  private ProximityManager(Context context, IKontaktCloud kontaktCloud) {
    checkNotNull(context, "Context can't be null").getApplicationContext();
    checkNotNull(kontaktCloud, "Kontakt Cloud can't be null");

    this.configuration = new Configuration(scanContext, internalProximityManagerConfiguration);
    this.internalProximityManager = new InternalProximityManager(
        context.getApplicationContext(),
        kontaktCloud,
        internalProximityManagerConfiguration
    );
  }

  @Override
  public void connect(OnServiceReadyListener onServiceReadyListener) {
    checkNotNull(onServiceReadyListener, "OnServiceReadyListener can't be null.");
    if (!internalProximityManager.isConnected()) {
      internalProximityManager.connect(onServiceReadyListener);
    } else {
      onServiceReadyListener.onServiceReady();
      Logger.d("BeaconManager is already connected.");
    }
  }

  @Override
  public void disconnect() {
    internalProximityManager.disconnect();
  }

  @Override
  public void startScanning() {
    checkArgument(internalProximityManager.isConnected(),
        "BeaconManager is not connected to ProximityService. Use BeaconManager.connect() before starting a scan.");
    if (!internalProximityManager.isScanning()) {
      createScanConfiguration();
      internalProximityManager.initializeScan(scanContext, internalProximityManagerConfiguration, createEventObserver());
    }
  }

  @Override
  public void stopScanning() {
    if (internalProximityManager.isScanning()) {
      internalProximityManager.finishScan();
    }
  }

  @Override
  public void restartScanning() {
    if (internalProximityManager.isScanning()) {
      createScanConfiguration();
      internalProximityManager.restartScan(scanContext, internalProximityManagerConfiguration, createEventObserver());
    }
  }

  @Override
  public boolean isConnected() {
    return internalProximityManager.isConnected();
  }

  @Override
  public boolean isScanning() {
    return internalProximityManager.isScanning();
  }

  @Override
  public void setScanStatusListener(ScanStatusListener listener) {
    this.scanStatusListener = listener;
  }

  @Override
  public void setSpaceListener(SpaceListener listener) {
    this.spaceListener = listener;
  }

  @Override
  public void setIBeaconListener(IBeaconListener listener) {
    this.iBeaconListener = listener;
  }

  @Override
  public void setEddystoneListener(EddystoneListener listener) {
    this.eddystoneListener = listener;
  }

  @Override
  public GeneralConfigurator configuration() {
    return configuration;
  }

  @Override
  public SpacesConfigurator spaces() {
    return configuration;
  }

  @Override
  public FiltersConfigurator filters() {
    return configuration;
  }

  private void createScanConfiguration() {
    checkObservedProfiles();
    scanContext = configuration.createScanContext();
    internalProximityManagerConfiguration = configuration.createKontaktManagerConfiguration();
  }

  private void checkObservedProfiles() {
    if (iBeaconListener == null) {
      configuration.removeObservedProfiles(DeviceProfile.IBEACON);
    } else {
      configuration.addObservedProfiles(DeviceProfile.IBEACON);
    }

    if (eddystoneListener == null) {
      configuration.removeObservedProfiles(DeviceProfile.EDDYSTONE);
    } else {
      configuration.addObservedProfiles(DeviceProfile.EDDYSTONE);
    }
  }

  private EventObserver createEventObserver() {
    return new EventObserver(eddystoneListener, iBeaconListener, scanStatusListener, spaceListener);
  }
}
