package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.bluetooth.BluetoothDevice;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.device.BeaconDevice;
import com.kontakt.sdk.android.ble.discovery.DiscoveryUtils;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.Parser;
import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import java.util.Arrays;
import java.util.Collection;
import java.util.UUID;

@SuppressWarnings("MissingPermission")
public final class IBeaconParser extends Parser<BeaconDevice> {

  public static final byte[] MANUFACTURER_DATA_IBEACON_PREFIX = new byte[] { 0x4C, 0x00, 0x02, 0x15 };
  private static final int MANUFACTURER_DATA_IBEACON_LENGTH = 25;

  private final Collection<IBeaconFilter> filters;

  public IBeaconParser(final ScanContext scanContext) {
    super(scanContext);
    this.filters = scanContext.getIBeaconFilters();
  }

  public boolean isValidIBeaconFrame(final byte[] scanRecord) {
    return ConversionUtils.doesArrayContainSubset(scanRecord, MANUFACTURER_DATA_IBEACON_PREFIX, 5);
  }

  public void parseScanRecord(final byte[] scanRecord) {
    frameData.clear();
    extractFrameData(scanRecord, frameData);
  }

  public boolean isManufacturerDataValid() {
    final byte[] manufacturerData = frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA);
    return manufacturerData.length == MANUFACTURER_DATA_IBEACON_LENGTH;
  }

  public boolean isScanResponsePresent() {
    byte[] scanResponseServiceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    return scanResponseServiceData != null && ScanResponse.isValidKontaktScanResponse(scanResponseServiceData);
  }

  public IBeaconDevice getIBeaconDevice(final BluetoothDevice device, final int rssi) {

    int deviceHashCode = hashCodeBuilder.append(device.getAddress())
        .append(frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA))
        .build();

    BeaconDevice iBeaconDevice = devicesCache.get(deviceHashCode);
    if (iBeaconDevice != null) {
      update(iBeaconDevice, deviceHashCode, rssi);
      return new BeaconDevice.Builder(iBeaconDevice).build();
    }

    byte[] manufacturerData = frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA);
    byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);

    final ScanResponse scanResponse = ScanResponse.fromScanResponseBytes(serviceData);
    final UUID proximityUUID = ConversionUtils.toUUID(Arrays.copyOfRange(manufacturerData, 4, 20));
    final int major = ConversionUtils.asInt(Arrays.copyOfRange(manufacturerData, 20, 22));
    final int minor = ConversionUtils.asInt(Arrays.copyOfRange(manufacturerData, 22, 24));
    final int calculatedRssi = rssiCalculator.calculateRssi(deviceHashCode, rssi);
    int txPower = manufacturerData[24];
    final double distance = DiscoveryUtils.calculateDistance(txPower, calculatedRssi, DeviceProfile.IBEACON);

    iBeaconDevice = new BeaconDevice.Builder().setAddress(device.getAddress())
        .setUniqueId(scanResponse.getUniqueId())
        .setBatteryPower(scanResponse.getBatteryPower())
        .setFirmwareVersion(scanResponse.getFirmwareVersion())
        .setShuffled(scanResponse.isShuffled())
        .setName(device.getName())
        .setProximityUUID(proximityUUID)
        .setMajor(major)
        .setMinor(minor)
        .setDistance(distance)
        .setProximity(Proximity.fromDistance(distance))
        .setTxPower(txPower)
        .setTimestamp(System.currentTimeMillis())
        .setRssi(calculatedRssi)
        .build();

    devicesCache.put(deviceHashCode, iBeaconDevice);

    return new BeaconDevice.Builder(iBeaconDevice).build();
  }

  public boolean filter(final IBeaconDevice device) {
    if (filters.isEmpty()) {
      return true;
    }

    for (final IBeaconFilter filter : filters) {
      if (filter.apply(device)) {
        return true;
      }
    }

    return false;
  }

  private void update(BeaconDevice device, int deviceHashCode, int rssi) {
    int calculatedRssi = rssiCalculator.calculateRssi(deviceHashCode, rssi);
    int txPower = frameData.get(FrameDataType.IBEACON_MANUFACTURER_SPECIFIC_DATA)[24];
    double distance = DiscoveryUtils.calculateDistance(txPower, calculatedRssi, DeviceProfile.IBEACON);

    device.setDistance(distance);
    device.setRssi(calculatedRssi);
    device.setProximity(Proximity.fromDistance(distance));
    device.setTimestamp(System.currentTimeMillis());
  }

  @Override
  protected void disable() {
    if (isEnabled) {
      isEnabled = false;
      filters.clear();
      devicesCache.clear();
      rssiCalculator.clear();
    }
  }
}
