package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.bluetooth.BluetoothDevice;
import android.util.SparseIntArray;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.device.EddystoneDevice;
import com.kontakt.sdk.android.ble.discovery.DiscoveryUtils;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.Parser;
import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.ble.filter.eddystone.EddystoneFilter;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.ble.spec.Telemetry;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.LimitedLinkedHashMap;
import java.util.Collection;
import java.util.Map;

@SuppressWarnings("MissingPermission")
public final class EddystoneParser extends Parser<EddystoneDevice> {

  private static final int HEADER_LENGTH = 12;

  private static final int HEADER_BLE_FLAGS_LENGTH_OF_FLAGS_BLOCK = 0x02;
  private static final int HEADER_BLE_FLAGS_PDU_DATA_TYPE = 0x01;
  private static final int HEADER_BLE_FLAGS_DATA = 0x06;
  private static final int HEADER_SERVICE_UUID_LENGTH_OF_SERVICE_UUID_BLOCK = 0x03;
  private static final int HEADER_SERVICE_UUID_PDU_DATA_TYPE = 0x03;

  private static final byte HEADER_SERVICE_UUID_HIGH_ORDER_VALUE = (byte) 0xAA;
  private static final byte HEADER_SERVICE_UUID_LOW_ORDER_VALUE = (byte) 0xFE;
  private static final int EDDYSTONE_PACKET_TX_POWER_INDEX = 3;

  private static final NamespaceIdResolver NAMESPACE_ID_RESOLVER = new NamespaceIdResolver();
  private static final InstanceIdResolver INSTANCE_ID_RESOLVER = new InstanceIdResolver();
  private static final URLResolver URL_RESOLVER = new URLResolver();
  private static final TLMResolver TLM_RESOLVER = new TLMResolver();

  private static byte[] EDDYSTONE_SPECIFIC_HEADER = new byte[] {
      HEADER_BLE_FLAGS_LENGTH_OF_FLAGS_BLOCK,
      HEADER_BLE_FLAGS_PDU_DATA_TYPE,
      HEADER_BLE_FLAGS_DATA,
      HEADER_SERVICE_UUID_LENGTH_OF_SERVICE_UUID_BLOCK,
      HEADER_SERVICE_UUID_PDU_DATA_TYPE,
      HEADER_SERVICE_UUID_HIGH_ORDER_VALUE,
      HEADER_SERVICE_UUID_LOW_ORDER_VALUE
  };

  private final Map<String, ScanResponse> scanResponseCache = new LimitedLinkedHashMap<>(CACHE_SIZE);
  private final SparseIntArray txPowerCache = new SparseIntArray();
  private final Collection<EddystoneFilter> filters;
  private final Collection<EddystoneFrameType> triggerFrameTypes;

  public EddystoneParser(final ScanContext scanContext) {
    super(scanContext);
    this.filters = scanContext.getEddystoneFilters();
    this.triggerFrameTypes = scanContext.getEddystoneFrameTypes();
  }

  public boolean isValidEddystoneFrame(final byte[] scanRecord) {
    return !(scanRecord == null || scanRecord.length < HEADER_LENGTH) &&
        ConversionUtils.doesArrayBeginWith(scanRecord, EDDYSTONE_SPECIFIC_HEADER) &&
        scanRecord[9] == HEADER_SERVICE_UUID_HIGH_ORDER_VALUE &&
        scanRecord[10] == HEADER_SERVICE_UUID_LOW_ORDER_VALUE;
  }

  public void parseFrame(EddystoneFrameType frameType, String deviceAddress, byte[] scanRecord) {
    frameData.clear();
    extractFrameData(scanRecord, frameData);
    byte[] packetFrame = frameData.get(FrameDataType.EDDYSTONE_PACKET_SERVICE_DATA);
    saveScanResponse(deviceAddress);
    switch (frameType) {
      case UID:
        saveUID(deviceAddress, packetFrame);
        break;
      case URL:
        saveURL(deviceAddress, packetFrame);
        break;
      case TLM:
        saveTLM(deviceAddress, packetFrame);
        break;
      default:
        Logger.e("Unknown Eddystone packetFrame type parsed for device with address: " + deviceAddress);
        break;
    }
  }

  void saveScanResponse(final String deviceAddress) {
    boolean isScanResponsePresent = isScanResponsePresent();
    byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    if (!scanResponseCache.containsKey(deviceAddress)) {
      ScanResponse scanResponse = isScanResponsePresent ? ScanResponse.fromScanResponseBytes(serviceData) : ScanResponse.UNKNOWN;
      scanResponseCache.put(deviceAddress, scanResponse);
    } else {
      if (scanResponseCache.get(deviceAddress).isUnknown() && isScanResponsePresent) {
        scanResponseCache.put(deviceAddress, ScanResponse.fromScanResponseBytes(serviceData));
      }
    }
  }

  void saveUID(String deviceAddress, byte[] frame) {
    updateTxPower(deviceAddress, frame[EDDYSTONE_PACKET_TX_POWER_INDEX]);

    String namespace = NAMESPACE_ID_RESOLVER.parse(frame);
    String instanceId = INSTANCE_ID_RESOLVER.parse(frame);

    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().setNamespace(namespace).setInstanceId(instanceId).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setNamespace(namespace);
      device.setInstanceId(instanceId);
    }
  }

  void saveURL(String deviceAddress, byte[] frame) {
    updateTxPower(deviceAddress, frame[EDDYSTONE_PACKET_TX_POWER_INDEX]);

    String url = URL_RESOLVER.parse(frame);

    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().setUrl(url).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setUrl(url);
    }
  }

  void saveTLM(String deviceAddress, byte[] frame) {
    Telemetry telemetry = TLM_RESOLVER.parse(frame);

    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().setTelemetry(telemetry).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setTelemetry(telemetry);
    }
  }

  IEddystoneDevice getEddystoneDevice(BluetoothDevice bluetoothDevice, int rssi) {
    String deviceAddress = bluetoothDevice.getAddress();

    int txPower = getTxPower(deviceAddress.hashCode());
    int calculatedRssi = rssiCalculator.calculateRssi(deviceAddress.hashCode(), rssi);
    double distance = DiscoveryUtils.calculateDistance(txPower, calculatedRssi, DeviceProfile.EDDYSTONE);
    Proximity proximity = Proximity.fromDistance(distance);

    String name = bluetoothDevice.getName();
    ScanResponse scanResponse = getScanResponse(deviceAddress);

    IEddystoneDevice cachedEddystoneDevice = getCachedEddystoneDevice(deviceAddress);
    EddystoneDevice device = new EddystoneDevice.Builder(cachedEddystoneDevice).setAddress(deviceAddress)
        .setName(name)
        .setUniqueId(scanResponse.getUniqueId())
        .setFirmwareVersion(scanResponse.getFirmwareVersion())
        .setBatteryPower(scanResponse.getBatteryPower())
        .setShuffled(scanResponse.isShuffled())
        .setTxPower(txPower)
        .setDistance(distance)
        .setProximity(proximity)
        .setRssi(calculatedRssi)
        .setTimestamp(System.currentTimeMillis())
        .build();

    putEddystoneDeviceInCache(deviceAddress, device);

    return device;
  }

  boolean filter(final String deviceAddress) {
    if (filters.isEmpty()) {
      return true;
    }

    final IEddystoneDevice eddystoneDevice = getCachedEddystoneDevice(deviceAddress);
    if (eddystoneDevice == null) {
      return false;
    }

    for (EddystoneFilter eddystoneFilter : filters) {
      if (eddystoneFilter.apply(eddystoneDevice)) {
        return true;
      }
    }

    return false;
  }

  public String getNamespaceForDevice(String deviceAddress) {
    return getCachedEddystoneDevice(deviceAddress).getNamespace();
  }

  public boolean areTriggerFramesParsed(String deviceAddress) {
    if (triggerFrameTypes.isEmpty()) {
      return true;
    }

    IEddystoneDevice eddystone = getCachedEddystoneDevice(deviceAddress);

    final boolean isUIDNotCached = (!hasFrame(eddystone, EddystoneFrameType.UID));
    if (triggerFrameTypes.contains(EddystoneFrameType.UID) && isUIDNotCached) {
      return false;
    }

    final boolean isURLNotCached = (!hasFrame(eddystone, EddystoneFrameType.URL));
    if (triggerFrameTypes.contains(EddystoneFrameType.URL) && isURLNotCached) {
      return false;
    }

    final boolean isTLMNotCached = (!hasFrame(eddystone, EddystoneFrameType.TLM));
    if (triggerFrameTypes.contains(EddystoneFrameType.TLM) && isTLMNotCached) {
      return false;
    }

    return true;
  }

  ScanResponse getScanResponse(String deviceAddress) {
    return scanResponseCache.get(deviceAddress);
  }

  int getTxPowerCacheSize() {
    return txPowerCache.size();
  }

  int getTxPower(int deviceAddressHashCode) {
    return txPowerCache.get(deviceAddressHashCode);
  }

  void putEddystoneDeviceInCache(String deviceAddress, EddystoneDevice device) {
    int hashCode = hashCodeBuilder.append(deviceAddress).build();
    devicesCache.put(hashCode, device);
  }

  EddystoneDevice getCachedEddystoneDevice(String deviceAddress) {
    int hashCode = hashCodeBuilder.append(deviceAddress).build();
    return devicesCache.get(hashCode);
  }

  private void updateTxPower(final String deviceAddress, final int txPower) {
    txPowerCache.put(deviceAddress.hashCode(), txPower);
  }

  boolean hasFrame(IEddystoneDevice eddystone, EddystoneFrameType frameType) {
    if (eddystone == null || frameType == null) {
      return false;
    }
    switch (frameType) {
      case UID:
        return eddystone.getNamespace() != null && eddystone.getInstanceId() != null;
      case URL:
        return eddystone.getUrl() != null;
      case TLM:
        return eddystone.getTelemetry() != null;
    }
    return false;
  }

  public boolean isScanResponsePresent() {
    byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    return serviceData != null && ScanResponse.isValidKontaktScanResponse(serviceData);
  }

  @Override
  protected void disable() {
    if(isEnabled) {
      isEnabled = false;
      filters.clear();
      devicesCache.clear();
      rssiCalculator.clear();
      scanResponseCache.clear();
      txPowerCache.clear();
    }
  }
}
