package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.bluetooth.BluetoothDevice;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.discovery.AbstractBluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import java.util.List;

public class EddystoneDiscoverer extends AbstractBluetoothDeviceDiscoverer<IEddystoneNamespace, IEddystoneDevice> {

  private final EddystoneParser parser;

  public EddystoneDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract) {
    super(discoveryContract, scanContext.getActivityCheckConfiguration(), scanContext.getEddystoneNamespaces(),
        scanContext.getDevicesUpdateCallbackInterval(), scanContext.isNonConnectableModeSupported());

    parser = new EddystoneParser(scanContext);
  }

  @Override
  protected BluetoothDeviceEvent createEvent(EventType eventType, IEddystoneNamespace eddystoneNamespace, List<IEddystoneDevice> deviceList) {
    return new EddystoneDeviceEvent(eventType, eddystoneNamespace, deviceList);
  }

  @Override
  protected void onBeforeDeviceLost(IEddystoneDevice device) {
    parser.clearRssiCalculation(device.getAddress().hashCode());
  }

  @Override
  public boolean performDiscovery(BluetoothDevice bluetoothDevice, int rssi, byte[] scanRecord) {
    String deviceAddress = bluetoothDevice.getAddress();

    if (parser.isEnabled() && !parser.isValidEddystoneFrame(scanRecord)) {
      return PROFILE_UNRECOGNIZED;
    }

    EddystoneFrameType frameType = EddystoneFrameType.fromScanRecord(scanRecord);
    if (frameType == null) {
      return PROFILE_UNRECOGNIZED;
    }

    parser.parseFrame(frameType, deviceAddress, scanRecord);
    notifyDevicePresent(deviceAddress.hashCode(), System.currentTimeMillis());

    if (!isNonConnectableModeSupported && !parser.isScanResponsePresent()) {
      return PROFILE_UNRECOGNIZED;
    }

    if (!parser.areTriggerFramesParsed(deviceAddress)) {
      return PROFILE_RECOGNIZED_FILTERING_NOT_PASSED;
    }

    IEddystoneDevice device = parser.getEddystoneDevice(bluetoothDevice, rssi);
    if(!parser.filter(deviceAddress)) {
      return PROFILE_RECOGNIZED_FILTERING_NOT_PASSED;
    }

    IEddystoneNamespace namespace = extractNamespace(parser.getNamespaceForDevice(deviceAddress));
    if (namespace == null && getSpaceSet().contains(EddystoneNamespace.EVERYWHERE)) {
      namespace = EddystoneNamespace.EVERYWHERE;
    } else if (namespace == null) {
      return PROFILE_RECOGNIZED_NO_BELONGING_SPACE_FOUND;
    }

    notifySpacePresent(namespace.hashCode(), System.currentTimeMillis());

    ReplacingArrayList<IEddystoneDevice> deviceList = getDevicesInSpace(namespace);
    if (deviceList == null) {
      deviceList = new ReplacingArrayList<>();
      insertDevicesIntoSpace(namespace, deviceList);
      onSpaceEnteredEvent(namespace);
    }

    if (deviceList.addOrReplace(device)) {
      onDeviceDiscoveredEvent(namespace, device);
    } else {
      onDevicesUpdatedEvent(namespace, deviceList);
    }

    return true;
  }

  @Override
  public void disable() {
    parser.disable();
  }

  IEddystoneNamespace extractNamespace(String inputNamespace) {
    for (IEddystoneNamespace namespace : getSpaceSet()) {
      if (namespace.getNamespace().equals(inputNamespace)) {
        return namespace;
      }
    }
    return null;
  }
}
