package com.kontakt.sdk.android.ble.discovery;

import android.annotation.TargetApi;
import android.os.Build;
import android.util.SparseLongArray;
import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.ble.util.SafeSparseLongArray;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Abstraction for Bluetooth device discoverers in the SDK. The abstraction provides
 * convenient interface whenever you need to implement your own Bluetooth device discoverer.
 *
 * @param <Space> Space type parameter which the discovered devices belong to
 * @param <Device> Bluetooth device type parameter
 */
@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public abstract class AbstractBluetoothDeviceDiscoverer<Space, Device extends RemoteBluetoothDevice> implements BluetoothDeviceDiscoverer {

  protected static boolean PROFILE_UNRECOGNIZED = false;
  protected static boolean PROFILE_RECOGNIZED_FILTERING_NOT_PASSED = false;
  protected static boolean PROFILE_RECOGNIZED_NO_BELONGING_SPACE_FOUND = false;

  protected final boolean isNonConnectableModeSupported;

  private final List<Device> EMPTY_DEVICE_LIST = new ArrayList<>();
  private final DiscoveryContract discoveryContract;
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final Collection<Space> spaceSet;
  private final Map<Space, ReplacingArrayList<Device>> spaceDeviceListMap = new ConcurrentHashMap<>();
  private final SparseLongArray spaceTimestampArray;
  private final SparseLongArray deviceTimestampArray;
  private final long devicesUpdateCallbackInterval;
  private long lastCallbackTime = 0;

  protected abstract BluetoothDeviceEvent createEvent(EventType eventType, Space space, List<Device> deviceList);

  protected abstract void onBeforeDeviceLost(Device device);

  protected AbstractBluetoothDeviceDiscoverer(DiscoveryContract discoveryContract, ActivityCheckConfiguration activityCheckConfiguration,
      Collection<Space> spaceSet, final long deviceUpdateCallbackInterval, boolean isNonConnectableModeSupported) {
    this.discoveryContract = discoveryContract;
    this.activityCheckConfiguration = activityCheckConfiguration;
    this.spaceSet = spaceSet;
    this.devicesUpdateCallbackInterval = deviceUpdateCallbackInterval;
    this.isNonConnectableModeSupported = isNonConnectableModeSupported;
    this.spaceTimestampArray = new SafeSparseLongArray(spaceSet.size());
    this.deviceTimestampArray = new SafeSparseLongArray();
  }

  @Override
  public void clearResources() {
    spaceTimestampArray.clear();
    deviceTimestampArray.clear();
    spaceDeviceListMap.clear();
  }

  protected void notifyDevicePresent(int deviceAddressHashcode, final long timestamp) {
    deviceTimestampArray.put(deviceAddressHashcode, timestamp);
  }

  protected void notifySpacePresent(final int spaceHashcode, final long timestamp) {
    spaceTimestampArray.put(spaceHashcode, timestamp);
  }

  protected Collection<Space> getSpaceSet() {
    return spaceSet;
  }

  protected ReplacingArrayList<Device> getDevicesInSpace(Space space) {
    return spaceDeviceListMap.get(space);
  }

  protected void insertDevicesIntoSpace(Space space, ReplacingArrayList<Device> deviceList) {
    spaceDeviceListMap.put(space, deviceList);
  }

  protected void onSpaceEnteredEvent(Space space) {
    discoveryContract.onEvent(createEvent(EventType.SPACE_ENTERED, space, EMPTY_DEVICE_LIST));
  }

  protected void onDeviceDiscoveredEvent(Space space, Device device) {
    final ArrayList<Device> deviceList = new ArrayList<>(1);
    deviceList.add(device);
    discoveryContract.onEvent(createEvent(EventType.DEVICE_DISCOVERED, space, deviceList));
  }

  protected void onDevicesUpdatedEvent(Space space, Collection<Device> deviceList) {
    final long currentTimeMillis = System.currentTimeMillis();
    if (lastCallbackTime + devicesUpdateCallbackInterval > currentTimeMillis) {
      return;
    }
    discoveryContract.onEvent(createEvent(EventType.DEVICES_UPDATE, space, new ArrayList<>(deviceList)));
    lastCallbackTime = currentTimeMillis;
  }

  protected void onDeviceLostEvent(Space space, Device device) {
    ArrayList<Device> deviceList = new ArrayList<>(1);
    deviceList.add(device);
    discoveryContract.onEvent(createEvent(EventType.DEVICE_LOST, space, deviceList));
  }

  protected void onSpaceAbandonedEvent(Space space) {
    discoveryContract.onEvent(createEvent(EventType.SPACE_ABANDONED, space, EMPTY_DEVICE_LIST));
  }

  @Override
  public void evictInactiveDevices(long currentTimeMillis) {
    final long inactivityTimeout = activityCheckConfiguration.getInactivityTimeout();
    for (Map.Entry<Space, ReplacingArrayList<Device>> entry : spaceDeviceListMap.entrySet()) {

      final Space space = entry.getKey();
      Iterator<Device> iterator = entry.getValue().iterator();
      while (iterator.hasNext()) {
        Device remoteDevice = iterator.next();
        long deviceTimestamp = deviceTimestampArray.get(remoteDevice.getAddress().hashCode(), -1L);
        if (deviceTimestamp != -1L) {
          long threshold = currentTimeMillis - deviceTimestamp;
          if (threshold > inactivityTimeout) {
            iterator.remove();
            onBeforeDeviceLost(remoteDevice);
            onDeviceLostEvent(space, remoteDevice);
            removeDeviceTimestamp(remoteDevice);
          }
        }
      }
    }

    evictInactiveRegions(inactivityTimeout, currentTimeMillis);
  }

  private void evictInactiveRegions(long inactivityTimeout, long currentTimeMillis) {
    final Set<Space> spaces = new HashSet<>(spaceDeviceListMap.keySet());

    for (final Space space : spaces) {
      final long regionTimestamp = spaceTimestampArray.get(space.hashCode(), -1L);
      if (regionTimestamp != -1L) {
        final long threshold = currentTimeMillis - regionTimestamp;
        if (threshold > inactivityTimeout) {
          spaceDeviceListMap.remove(space);
          onSpaceAbandonedEvent(space);
          removeSpaceTimestamp(space);
        }
      }
    }
  }

  private void removeDeviceTimestamp(Device remoteDevice) {
    int index = deviceTimestampArray.indexOfKey(remoteDevice.getAddress().hashCode());
    if (index >= 0) {
      deviceTimestampArray.removeAt(index);
    }
  }

  private void removeSpaceTimestamp(Space space) {
    int index = spaceTimestampArray.indexOfKey(space.hashCode());
    if (index >= 0) {
      spaceTimestampArray.removeAt(index);
    }
  }

}
