package com.kontakt.sdk.android.ble.connection;

/**
 * WriteListener provides callbacks indicating whether Beacon characteristics manipulation
 * ended successfully or failed.
 */
public interface WriteListener {

  /**
   * Invoked when characteristic manipulation suceed.
   * <br>
   * If there is extra response from device it will be passed as {@link com.kontakt.sdk.android.ble.connection.WriteListener.WriteResponse} response
   *
   * @param response from device, might be null
   */
  void onWriteSuccess(WriteResponse response);

  /**
   * Invoked when characteristics manipulation failed.
   *
   * @param cause {@link Cause}
   */
  void onWriteFailure(Cause cause);

  /**
   * The NULL Listener is set when the reference to WriteListener in Overwriting methods
   * is no longer needed.
   */
  WriteListener NULL_LISTENER = new WriteListener() {
    @Override
    public void onWriteSuccess(WriteResponse response) {
    }

    @Override
    public void onWriteFailure(Cause cause) {

    }
  };

  enum Cause {
    GATT_FAILURE,
    FEATURE_NOT_SUPPORTED,
    INCORRECT_VALUE
  }

  /**
   * Object representing response from device after writing characteristic
   */
  class WriteResponse {
    private final long unixTimestamp;
    private final String extra;

    public WriteResponse(long unixTimestamp, String extra) {
      this.unixTimestamp = unixTimestamp;
      this.extra = extra;
    }

    /**
     * Method for getting unix based timestamp of successful beacon characteristic manipulation
     *
     * @return unixTimestamp
     */
    public long getUnixTimestamp() {
      return unixTimestamp;
    }

    /**
     * Gets extra message from beacon device
     *
     * @return extra message
     */
    public String getExtra() {
      return extra;
    }

    /**
     * <p>
     * Method for checking is there extra message from beacon device
     * <br>
     * Always true for beacon with firmware version 4.0 and higher
     *
     *
     * @return true if beacon responded with message
     */
    public boolean hasExtra() {
      return extra != null;
    }

    @Override
    public String toString() {
      return "WriteResponse{" +
          "unixTimestamp=" + unixTimestamp +
          ", extra='" + extra + '\'' +
          '}';
    }
  }
}