package com.kontakt.sdk.android.ble.configuration.scan;

import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.device.BeaconRegion;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.filter.eddystone.EddystoneFilter;
import com.kontakt.sdk.android.ble.filter.ibeacon.IBeaconFilter;
import com.kontakt.sdk.android.ble.rssi.RssiCalculator;
import com.kontakt.sdk.android.ble.rssi.RssiCalculators;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Provides scan-specific parameters influencing scan performance
 * profiles of BLE devices to be found.
 */
public final class ScanContext implements GlobalScanContext {

  /**
   * The constant DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL.
   */
  public static final long DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL = 3000;

  /**
   * Default settings which works well on most modern devices
   */
  public static final ScanContext DEFAULT = new ScanContext.Builder().build();

  private final ForceScanConfiguration forceScanConfiguration;
  private final ScanPeriod scanPeriod;
  private final ScanMode scanMode;
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final long devicesUpdateCallbackInterval;
  private final RssiCalculator rssiCalculator;
  private final boolean supportNonConnectable;
  private final Set<DeviceProfile> observedProfiles;
  private final Set<EddystoneFrameType> eddystoneTriggerFrameTypes;
  private final Set<IBeaconRegion> iBeaconRegions;
  private final Set<IEddystoneNamespace> eddystoneNamespaces;
  private final List<IBeaconFilter> iBeaconFilters;
  private final List<EddystoneFilter> eddystoneFilters;

  private ScanContext(Builder builder) {
    this.forceScanConfiguration = builder.forceScanConfiguration;
    this.scanPeriod = builder.scanPeriod;
    this.scanMode = builder.scanMode;
    this.activityCheckConfiguration = builder.activityCheckConfiguration;
    this.devicesUpdateCallbackInterval = builder.devicesUpdateCallbackInterval;
    this.rssiCalculator = builder.rssiCalculator;
    this.supportNonConnectable = builder.supportNonConnectable;
    this.observedProfiles = builder.observedProfiles;
    this.eddystoneTriggerFrameTypes = builder.eddystoneTriggerFrameTypes;
    this.iBeaconRegions = builder.iBeaconRegions;
    this.eddystoneNamespaces = builder.eddystoneNamespaces;
    this.iBeaconFilters = builder.iBeaconFilters;
    this.eddystoneFilters = builder.eddystoneFilters;
  }

  @Override
  public ActivityCheckConfiguration getActivityCheckConfiguration() {
    return activityCheckConfiguration;
  }

  @Override
  public Set<DeviceProfile> getObservedProfiles() {
    return observedProfiles;
  }

  @Override
  public ForceScanConfiguration getForceScanConfiguration() {
    return forceScanConfiguration;
  }

  @Override
  public ScanPeriod getScanPeriod() {
    return scanPeriod;
  }

  @Override
  public ScanMode getScanMode() {
    return scanMode;
  }

  @Override
  public long getDevicesUpdateCallbackInterval() {
    return devicesUpdateCallbackInterval;
  }

  @Override
  public RssiCalculator getRssiCalculator() {
    return rssiCalculator;
  }

  @Override
  public boolean isNonConnectableModeSupported() {
    return supportNonConnectable;
  }

  @Override
  public Collection<EddystoneFrameType> getEddystoneFrameTypes() {
    return eddystoneTriggerFrameTypes;
  }

  @Override
  public Set<IBeaconRegion> getIBeaconRegions() {
    return iBeaconRegions;
  }

  @Override
  public Set<IEddystoneNamespace> getEddystoneNamespaces() {
    return eddystoneNamespaces;
  }

  @Override
  public List<IBeaconFilter> getIBeaconFilters() {
    return iBeaconFilters;
  }

  @Override
  public List<EddystoneFilter> getEddystoneFilters() {
    return eddystoneFilters;
  }

  public static final class Builder {
    private boolean supportNonConnectable;
    private ScanMode scanMode = ScanMode.BALANCED;
    private ScanPeriod scanPeriod = ScanPeriod.RANGING;
    private ForceScanConfiguration forceScanConfiguration = ForceScanConfiguration.DISABLED;
    private ActivityCheckConfiguration activityCheckConfiguration = ActivityCheckConfiguration.DEFAULT;
    private RssiCalculator rssiCalculator = RssiCalculators.DEFAULT;
    private long devicesUpdateCallbackInterval = DEFAULT_DEVICES_UPDATE_CALLBACK_INTERVAL;
    private Set<DeviceProfile> observedProfiles = EnumSet.allOf(DeviceProfile.class);
    private Set<EddystoneFrameType> eddystoneTriggerFrameTypes = EnumSet.noneOf(EddystoneFrameType.class);
    private Set<IBeaconRegion> iBeaconRegions = new HashSet<>();
    private Set<IEddystoneNamespace> eddystoneNamespaces = new HashSet<>();
    private List<IBeaconFilter> iBeaconFilters = new ArrayList<>();
    private List<EddystoneFilter> eddystoneFilters = new ArrayList<>();

    public Builder() {

    }

    public Builder setActivityCheckConfiguration(ActivityCheckConfiguration activityCheckConfiguration) {
      SDKPreconditions.checkNotNull(activityCheckConfiguration, "Beacon activity check is null.");
      this.activityCheckConfiguration = activityCheckConfiguration;
      return this;
    }

    public Builder setScanMode(ScanMode scanMode) {
      SDKPreconditions.checkNotNull(scanMode, "Scan mode is null");
      this.scanMode = scanMode;
      return this;
    }

    public Builder setForceScanConfiguration(ForceScanConfiguration forceScanConfiguration) {
      SDKPreconditions.checkNotNull(forceScanConfiguration, "By default ForceScanConfiguration is disabled");
      this.forceScanConfiguration = forceScanConfiguration;
      return this;
    }

    public Builder setScanPeriod(ScanPeriod scanPeriod) {
      SDKPreconditions.checkNotNull(scanPeriod, "Monitor period cannot be null");
      this.scanPeriod = scanPeriod;
      return this;
    }

    public Builder setDevicesUpdateCallbackInterval(long intervalInMillis) {
      SDKPreconditions.checkArgument(intervalInMillis > 0, "Interval must be greater than 0");
      this.devicesUpdateCallbackInterval = intervalInMillis;
      return this;
    }

    public Builder setRssiCalculator(RssiCalculator rssiCalculator) {
      SDKPreconditions.checkNotNull(rssiCalculator, "RssiCalculator can't be null");
      this.rssiCalculator = rssiCalculator;
      return this;
    }

    public Builder setSupportNonConnectableMode(boolean enabled) {
      this.supportNonConnectable = enabled;
      return this;
    }

    public Builder setObservedProfiles(Set<DeviceProfile> observedProfiles) {
      SDKPreconditions.checkNotNull(observedProfiles, "Observed profiles can't be null");
      this.observedProfiles = observedProfiles;
      return this;
    }

    public Builder setIBeaconFilters(Collection<? extends IBeaconFilter> filters) {
      SDKPreconditions.checkNotNull(filters, "Filters are null.");
      this.iBeaconFilters.clear();
      this.iBeaconFilters.addAll(filters);
      return this;
    }

    public Builder setIBeaconRegions(final Collection<IBeaconRegion> iBeaconRegions) {
      SDKPreconditions.checkNotNull(iBeaconRegions, "Regions collection is null");
      this.iBeaconRegions.clear();
      this.iBeaconRegions.addAll(iBeaconRegions);
      return this;
    }

    public Builder setEddystoneFilters(Collection<EddystoneFilter> eddystoneFilters) {
      SDKPreconditions.checkNotNull(eddystoneFilters, "Filters are null.");
      this.eddystoneFilters.clear();
      this.eddystoneFilters.addAll(eddystoneFilters);
      return this;
    }

    public Builder setEddystoneNamespaces(Collection<IEddystoneNamespace> eddystoneNamespaces) {
      SDKPreconditions.checkNotNull(eddystoneNamespaces, "Eddystone namespaces are null.");
      this.eddystoneNamespaces.clear();
      this.eddystoneNamespaces.addAll(eddystoneNamespaces);
      return this;
    }

    public Builder setRequiredEddystoneFrameTypes(Collection<EddystoneFrameType> eddystoneFrames) {
      SDKPreconditions.checkNotNull(eddystoneFrames, "Eddystone trigger frames");
      this.eddystoneTriggerFrameTypes.clear();
      this.eddystoneTriggerFrameTypes.addAll(eddystoneFrames);
      return this;
    }

    public ScanContext build() {
      SDKPreconditions.checkNotNullOrEmpty(observedProfiles, "At least one listener (IBeacon or Eddystone) must be set.");
      ScanContextValidator.validate(scanPeriod);
      ScanContextValidator.validateEddystoneFiltersCount(eddystoneFilters);
      ScanContextValidator.validateIBeaconFiltersCount(iBeaconFilters);
      ScanContextValidator.validateIBeaconRegionsCount(iBeaconRegions);
      ScanContextValidator.validateNamespacesCount(eddystoneNamespaces);
      ScanContextValidator.validate(activityCheckConfiguration);
      ScanContextValidator.validate(forceScanConfiguration);
      ScanContextValidator.validate(activityCheckConfiguration, scanPeriod);

      if (iBeaconRegions.isEmpty()) {
        iBeaconRegions.add(BeaconRegion.EVERYWHERE);
      }
      if (eddystoneNamespaces.isEmpty()) {
        eddystoneNamespaces.add(EddystoneNamespace.EVERYWHERE);
      }

      return new ScanContext(this);
    }
  }
}
