package com.kontakt.sdk.android.ble.configuration;

import com.kontakt.sdk.android.ble.manager.internal.InternalProximityManager;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.regex.Pattern;

/**
 * Represents configuration for {@link InternalProximityManager}.
 * <p>
 * Parameters:
 *
 * <ul>
 * <li> cacheFileName - cache file name for resolved shuffled device ids from Kontakt.io Proximity REST API.
 * Cache file name must be unique for safe serialization between different {@link InternalProximityManager} used in the same app</li>
 * <li> resolveShuffledInterval - interval between next resolving attempts in seconds. Must be greater than 0. </li>
 * <li> monitoringEnabled - enables or disables collecting monitoring events e.g. battery level of device.</li>
 * <li> monitoringSyncInterval - interval between next monitoring events sync attempts in seconds. Must be greater than 0.</li>
 * </ul>
 * Resolve delay and buffer size concern all of supported protocols (iBeacon and Eddystone) separately. Cache file is common for all protocols.
 */
public final class InternalProximityManagerConfiguration {

  private static final int MIN_RESOLVE_INTERVAL = 0;
  private static final int MIN_MONITORING_SYNC_INTERVAL = 0;
  private static final String FILE_NAME_REGEX = "^[^(^?|*<\":>+\\[\\]\\/'{})]+$";
  private static final Pattern FILE_NAME_MATCHER = Pattern.compile(FILE_NAME_REGEX);

  /**
   * Default cache file name.
   */
  public static final String DEFAULT_CACHE_FILE_NAME = "resolved.che";

  /**
   * Default resolve interval in seconds.
   */
  public static final int DEFAULT_RESOLVE_INTERVAL = 3;

  /**
   * Default monitoring sync interval in seconds.
   */
  public static final int DEFAULT_MONITORING_SYNC_INTERVAL = 10;

  /**
   * Default monitoring enabled
   */
  public static final boolean DEFAULT_MONITORING_ENABLED = true;

  /**
   * Default configuration for {@link InternalProximityManager}
   */
  public static final InternalProximityManagerConfiguration DEFAULT = new Builder()
      .setCacheFileName(DEFAULT_CACHE_FILE_NAME)
      .setResolveInterval(DEFAULT_RESOLVE_INTERVAL)
      .setMonitoringSyncInterval(DEFAULT_MONITORING_SYNC_INTERVAL)
      .setMonitoringEnabled(DEFAULT_MONITORING_ENABLED)
      .build();

  private final String cacheFileName;
  private final int resolveInterval;
  private final int monitoringSyncInterval;
  private final boolean monitoringEnabled;

  private InternalProximityManagerConfiguration(Builder builder) {
    this.cacheFileName = builder.cacheFileName;
    this.resolveInterval = builder.resolveInterval;
    this.monitoringSyncInterval = builder.monitoringSyncInterval;
    this.monitoringEnabled = builder.monitoringEnabled;
  }

  public String getCacheFileName() {
    return cacheFileName;
  }

  public int getResolveInterval() {
    return resolveInterval;
  }

  public int getMonitoringSyncInterval() {
    return monitoringSyncInterval;
  }

  public boolean isMonitoringEnabled() {
    return monitoringEnabled;
  }

  /**
   * Configuration builder is responsible for creating immutable configuration for {@link InternalProximityManager}.
   * <br>
   * By default the Builder is initialized with all default parameters.
   *
   * @see InternalProximityManagerConfiguration
   */
  public static class Builder {
    private String cacheFileName = DEFAULT_CACHE_FILE_NAME;
    private int resolveInterval = DEFAULT_RESOLVE_INTERVAL;
    private boolean monitoringEnabled = DEFAULT_MONITORING_ENABLED;
    private int monitoringSyncInterval = DEFAULT_MONITORING_SYNC_INTERVAL;

    public Builder setCacheFileName(String cacheFileName) {
      SDKPreconditions.checkNotNull(cacheFileName, "Cache file name cannot be null!");
      boolean match = FILE_NAME_MATCHER.matcher(cacheFileName).matches();
      SDKPreconditions.checkArgument(match, "Incorrect cache file name");
      this.cacheFileName = cacheFileName;
      return this;
    }

    public Builder setResolveInterval(int resolveIntervalSeconds) {
      SDKPreconditions.checkArgument(resolveIntervalSeconds > MIN_RESOLVE_INTERVAL, "Resolve interval must be greater than 0!");
      this.resolveInterval = resolveIntervalSeconds;
      return this;
    }

    public Builder setMonitoringSyncInterval(int monitoringSyncIntervalSeconds) {
      SDKPreconditions.checkArgument(monitoringSyncIntervalSeconds > MIN_MONITORING_SYNC_INTERVAL,
          "Monitoring sync interval must be greater than 0!");
      this.monitoringSyncInterval = monitoringSyncIntervalSeconds;
      return this;
    }

    public Builder setMonitoringEnabled(boolean monitoringEnabled) {
      this.monitoringEnabled = monitoringEnabled;
      return this;
    }

    public InternalProximityManagerConfiguration build() {
      return new InternalProximityManagerConfiguration(this);
    }
  }
}
