package com.kontakt.sdk.android.ble.configuration;

/**
 * According to the bug no 65863 reported at Android Open Source Project
 * some Android devices perform single scan for each BLE device found.
 * We served this bug by indtroducing configuration which will adjust the
 * performance of the SDK to device that it will be working on. During this
 * mode a Bluetooth Adapter
 * ({@link android.bluetooth.BluetoothAdapter}) performs scan restart
 * periodically.
 *
 * There are 2 predefined constants:
 * <ul>
 * <li>DISABLED - indicates that ForceScanConfiguration should not be
 * enabled because Android device detects BLE devices after single launch</li>
 * <li>DEFAULT - provides default ForceScanConfiguration with Active Scan Period
 * equal to 1 s (BLE scan is enabled) and Passive Scan Period 1 s
 * (BLE scan is disabled).</li>
 * </ul>
 *
 * Please note that parameters specified in MINIMAL predefined constant are the
 * lowest possible. Any attempt of setting configuration with lower parameters
 * will be validated with IllegalArgumentException.
 *
 *
 * It is advisable to make SDK-based application configurable per
 * Android device e.g. by allowing changing ForceScanConfiguration
 * parameters in preferences. Ignoring it may cause poor performance of
 * features provided for the SDK.
 *
 * @see <a href="https://code.google.com/p/android/issues/detail?id=65863" target="_blank">Android Open Source Project - bug no 65863</a>
 */
public class ForceScanConfiguration {

  private final long forceScanActivePeriod;
  private final long forceScanPassivePeriod;

  /**
   * Applying this predefined constant causes scan forcing to be disabled.
   */
  public static final ForceScanConfiguration DISABLED = new ForceScanConfiguration(0L, 0L);

  /**
   * Predefined default Force Scan Configuration. The default Force Scan
   * Configuration contains parameters with following values:
   *
   * <ul>
   * <li>forceScanActivePeriod is set to 1s. During this period device
   * will be scanning BLE devices. Please note that you cannot specify
   * active period shorter than default one.
   * </li>
   * <li>forceScanPassivePeirod is set to 500ms. During this period device
   * will not be scanning BLE devices. Please not that you cannot specify
   * passive period shorter than default one.
   * </li>
   * </ul>
   */
  public static final ForceScanConfiguration MINIMAL = new ForceScanConfiguration(1000, 500);

  /**
   * Instantiates a new Force Scan Configuration.
   *
   * @param forceScanActivePeriod the force scan active period [ms]
   * @param forceScanPassivePeriod the force scan passive period [ms]
   */
  public ForceScanConfiguration(long forceScanActivePeriod, long forceScanPassivePeriod) {
    this.forceScanActivePeriod = forceScanActivePeriod;
    this.forceScanPassivePeriod = forceScanPassivePeriod;
  }

  /**
   * Provides force scan active period (in milliseconds).
   *
   * @return the force scan active period
   */
  public long getForceScanActivePeriod() {
    return forceScanActivePeriod;
  }

  /**
   * Provides force scan passive period (in milliseconds).
   *
   * @return the force scan passive period
   */
  public long getForceScanPassivePeriod() {
    return forceScanPassivePeriod;
  }
}
