package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.ble.device.BeaconDevice;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconDeviceEvent;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.IBeaconFutureId;
import com.kontakt.sdk.android.common.model.IBeaconId;
import com.kontakt.sdk.android.common.model.ResolvedId;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;

class IBeaconIdResolver implements Runnable {

  private static final String TAG = "IBeaconIdResolver";
  private static final int REQUEST_UNIT_SIZE = 70;
  private static final int DEFAULT_BUFFER_SIZE = 200;
  private static final DeviceProfile DEVICE_PROFILE = DeviceProfile.IBEACON;
  private final ArrayBlockingQueue<IBeaconResolveRequest> requestQueue;
  private final FutureShufflesCache cache;
  private final Collection<IBeaconResolveStrategy> strategies;

  IBeaconIdResolver(FutureShufflesCache futureShufflesCache, IKontaktCloud kontaktCloud) {
    this.cache = futureShufflesCache;
    this.requestQueue = new ArrayBlockingQueue<>(DEFAULT_BUFFER_SIZE, true);
    List<IBeaconResolveStrategy> resolvers = new LinkedList<>();
    IBeaconResolveStrategy cacheStrategy = new IBeaconCacheResolveStrategy(cache);
    IBeaconResolveStrategy apiStrategy = new IBeaconApiResolveStrategy(kontaktCloud);
    resolvers.add(cacheStrategy);
    resolvers.add(apiStrategy);
    this.strategies = Collections.unmodifiableCollection(resolvers);
  }

  @Override public void run() {
    if (!cache.isInitialized()) {
      Logger.d(TAG + " Cache not initialized yet");
      return;
    }
    final List<IBeaconResolveRequest> requests = new ArrayList<>();
    requestQueue.drainTo(requests, REQUEST_UNIT_SIZE);
    if (requests.isEmpty()) {
      Logger.d(TAG + " Nothing to resolve");
      return;
    }
    try {
      Logger.d(TAG + " Start resolving");

      final Map<IBeaconId, IBeaconResolveRequest> requestsRegister = buildRequestsRegister(requests);
      final List<IBeaconFutureId> shuffles = resolveShuffles(requestsRegister);
      final Map<IBeaconId, IBeaconFutureId> shufflesRegister = buildShufflesRegister(shuffles);

      evictOutdatedCacheEntries(requestsRegister, shufflesRegister);
      handleRequests(requestsRegister, shufflesRegister);

      cache.serialize();
    } catch (Exception e) {
      Throwable cause = e.getCause();
      if (UnknownHostException.class.isInstance(cause) || SocketTimeoutException.class.isInstance(cause)) {
        requestQueue.addAll(requests);
      } else {
        Logger.e(TAG + " Error occurs when try to resolve shuffled device ", e);
      }
    }
  }

  void addResolveRequest(IBeaconResolveRequest request) {
    if (requestQueue.contains(request)) {
      return;
    }
    try {
      requestQueue.add(request);
    } catch (IllegalStateException fullQueueException) {
      Logger.e("Could not add iBeacon to resolve", fullQueueException);
    }
  }

  public void markIgnored(RemoteBluetoothDevice beacon) {
    for (IBeaconResolveRequest request : requestQueue) {
      if (request.getFakeDevice().equals(beacon)) {
        request.setStatus(ResolveRequestStatus.IGNORED);
      }
    }
  }

  public void clear() {
    requestQueue.clear();
  }

  private Map<IBeaconId, IBeaconResolveRequest> buildRequestsRegister(final List<IBeaconResolveRequest> requests) {
    final Map<IBeaconId, IBeaconResolveRequest> requestsRegister = new HashMap<>();
    for (IBeaconResolveRequest request : requests) {
      IBeaconId beaconId = IBeaconId.fromDevice(request.getFakeDevice());
      requestsRegister.put(beaconId, request);
    }
    return requestsRegister;
  }

  private List<IBeaconFutureId> resolveShuffles(final Map<IBeaconId, IBeaconResolveRequest> requestsRegister) throws Exception {
    final List<IBeaconFutureId> shuffles = new ArrayList<>();
    for (IBeaconResolveStrategy strategy : strategies) {
      List<IBeaconFutureId> resolved = strategy.resolve(requestsRegister);
      shuffles.addAll(resolved);
    }
    return shuffles;
  }

  private Map<IBeaconId, IBeaconFutureId> buildShufflesRegister(final List<IBeaconFutureId> shuffles) {
    final Map<IBeaconId, IBeaconFutureId> shufflesRegister = new HashMap<>();
    for (IBeaconFutureId futureShuffle : shuffles) {
      shufflesRegister.put(futureShuffle.getQueriedBy(), futureShuffle);
    }
    return shufflesRegister;
  }

  private void evictOutdatedCacheEntries(final Map<IBeaconId, IBeaconResolveRequest> requestsRegister,
      final Map<IBeaconId, IBeaconFutureId> shufflesRegister) {
    final List<String> uniqueIds = new ArrayList<>();
    for (Map.Entry<IBeaconId, IBeaconResolveRequest> entry : requestsRegister.entrySet()) {
      IBeaconId queriedBy = entry.getKey();
      IBeaconResolveRequest request = entry.getValue();
      IBeaconFutureId iBeaconShuffles = shufflesRegister.get(queriedBy);
      if (iBeaconShuffles == null || ResolverType.CACHE == request.getResolvedBy()) {
        continue;
      }
      // non null shuffles resolved by api
      uniqueIds.add(iBeaconShuffles.getUniqueId());
    }

    // remove old (OLD_DEVICE_ID -> RESOLVED_ID) entries from cache and
    // replace them with the new ones (NEW_DEVICE_ID -> RESOLVED_ID) in next step (@addNewCacheEntries)
    cache.evict(uniqueIds, DEVICE_PROFILE);
  }

  private void handleRequests(final Map<IBeaconId, IBeaconResolveRequest> requestsRegister,
      final Map<IBeaconId, IBeaconFutureId> shufflesRegister) {
    for (Map.Entry<IBeaconId, IBeaconResolveRequest> entry : requestsRegister.entrySet()) {
      handleRequest(shufflesRegister, entry);
    }
  }

  private void handleRequest(final Map<IBeaconId, IBeaconFutureId> shufflesRegister,
      final Map.Entry<IBeaconId, IBeaconResolveRequest> entry) {
    final IBeaconId queriedBy = entry.getKey();
    final IBeaconFutureId iBeaconShuffles = shufflesRegister.get(queriedBy);
    if (iBeaconShuffles == null) {
      cache.populate(queriedBy.toString(), FutureShufflesCache.PHANTOM_ENTRY);
      return;
    }

    final IBeaconResolveRequest request = entry.getValue();
    final IBeaconId resolvedId = iBeaconShuffles.getResolved();

    final String uniqueId = iBeaconShuffles.getUniqueId();
    final ResolvedId resolvedBeaconId = ResolvedId.create(resolvedId.toString(), uniqueId, DEVICE_PROFILE);

    // after remove old entries from cache (@evictOutdatedCacheEntries), add the new ones
    if (ResolverType.CACHE != request.getResolvedBy()) {
      addNewCacheEntries(iBeaconShuffles, resolvedBeaconId);
    }

    if (ResolveRequestStatus.RESOLVED == request.getStatus()) {
      notifyListeners(request, resolvedBeaconId);
    }
  }

  private void addNewCacheEntries(final IBeaconFutureId iBeaconShuffles, final ResolvedId resolvedBeaconId) {
    for (IBeaconId futureShuffle : iBeaconShuffles.getFutureIds()) {
      cache.populate(futureShuffle.toString(), resolvedBeaconId);
    }
    cache.populate(resolvedBeaconId.getIBeaconId().toString(), resolvedBeaconId);
  }

  private void notifyListeners(final IBeaconResolveRequest request, final ResolvedId resolvedBeaconId) {
    final IBeaconDevice fakeDevice = request.getFakeDevice();
    final IBeaconDevice resolvedDevice = BeaconDevice.of(fakeDevice, resolvedBeaconId);
    final Integer proximityManagerId = request.getSourceProximityManagerId();
    final BluetoothDeviceEvent deviceEvent = IBeaconDeviceEvent.createNewDiscovered(
        resolvedDevice, request.getRegion(), request.getTimestamp());
    cache.notifyListeners(proximityManagerId, deviceEvent);
  }
}
