package com.kontakt.sdk.android.common.util;

import android.text.TextUtils;
import com.kontakt.sdk.android.common.FirmwareRevisions;

/**
 * Provides checking methods for different properties of iBeacon devices.
 *
 * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
 */
public final class IBeaconPropertyValidator {

    private IBeaconPropertyValidator() {
    }

    private static final int MAX_MAJOR = Double.valueOf(Math.pow(2, 16)).intValue();

    private static final int MAX_MINOR = Double.valueOf(Math.pow(2, 16)).intValue();
    private static final int PASSWORD_LENGTH = 4;
    private static final int PASSWORD_MAX_LENGTH = 16;

    private static final int MASTER_PASSWORD_MAX_LENGTH = 6;

    private static final int MODEL_NAME_MAX_LENGTH = 16;

    public static void validateModelName(final String name) {
        SDKPreconditions.checkArgument(!TextUtils.isEmpty(name), "Model name is null or empty");
        SDKPreconditions.checkArgument(name.length() < MODEL_NAME_MAX_LENGTH, "Device name length must not be higher than 15 ASCII symbols.");
        EddystoneUtils.isAllASCIIPrintable(name);
    }

    /**
     * Validates Beacon major value the value should be within the range [0, 65535].
     *
     * @param major the major
     */
    public static void validateMajor(final int major) {
        SDKPreconditions.checkArgument(major >= 0 && major < MAX_MAJOR,
                "Allowed range of major value: [1...65535]");
    }

    /**
     * Validates Beacon minor value the value should be within the range [0, 65535]
     *
     * @param minor the minor
     */
    public static void validateMinor(final int minor) {
        SDKPreconditions.checkArgument(minor >= 0 && minor < MAX_MINOR,
                "Allowed range of minor value: [1...65535]");
    }

    /**
     * Validates Beacon power level.
     *
     * @param txPowerLevel the tx power level
     * @see <a href="docs.kontakt.io/beacon/kontakt-beacon-v2.pdf">kontakt.io Beacon Datasheet - version 2.0</a>
     */
    public static void validatePowerLevel(final int txPowerLevel) {
        ConversionUtils.convertPowerLevel(txPowerLevel);
    }

    /**
     * Validates Beacon advertising interval. The interval value should be within [20...10240] ms.
     *
     * @param beaconIntervalMillis the beacon interval millis
     * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
     */
    public static void validateAdvertisingInterval(final int beaconIntervalMillis) {
        SDKPreconditions.checkArgument(beaconIntervalMillis >= 20 && beaconIntervalMillis <= 10240,
                "Advertising interval must be within range: [20...10240] ms.");
    }

    /**
     * Validates Beacon password. <br>
     * The password should be 4-byte long (4 ASCII characters) for Beacons with firmware version lower than 4.0. <br>
     * The password should be 1 to 16 byte long (1 to 16 ASCII characters) for Beacons with firmware version 4.0 or higher
     *
     * @param beaconPassword the beacon password
     * @param firmwareRevision firmware revision
     * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
     */
    public static void validateBeaconPassword(final String beaconPassword, final String firmwareRevision) {
        SDKPreconditions.checkNotNullOrEmpty(beaconPassword, "New Password is null or empty.");
        if(FirmwareRevisions.NORMAL_MODE_FIRMWARE_VERSIONS.contains(firmwareRevision)) {
            SDKPreconditions.checkArgument(beaconPassword.length() == PASSWORD_LENGTH,
                "Beacon password must consist of 4 ASCII symbols");
        } else if (FirmwareRevisions.SECURE_MODE_FIRMWARE_VERSIONS.contains(firmwareRevision)) {
            SDKPreconditions.checkArgument(beaconPassword.length() >= PASSWORD_LENGTH && beaconPassword.length() <= PASSWORD_MAX_LENGTH,
                "Beacon password must consist of 4 to 16 ASCII symbols");
        }
    }

    /**
     * Validates beacon master password. The master password should be 6-byte long (6 ASCII characters).
     *
     * @param masterPassword the master password
     * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0</a>
     */
    public static void validateBeaconMasterPassword(final String masterPassword) {
        SDKPreconditions.checkArgument(masterPassword.length() == MASTER_PASSWORD_MAX_LENGTH,
                "Beacon master password must consist of 6 ASCII symbols");
    }
}
