package com.kontakt.sdk.android.common.model;

import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.Serializable;
import java.util.UUID;

/**
 * This class is a connection between device ID in Eddystone (namespace:instanceId) or iBeacon
 * (proximity:major:minor) format and an unique ID of {@link Device} object. It can be useful while
 * resolving shuffled devices.
 */
public class ResolvedId implements Serializable {

  private static final long serialVersionUID = 1L;
  private static final String ID_SEPARATOR = ":";
  private final String deviceId;
  private final String uniqueId;
  private Integer hashCode;

  /**
   * Creates a connection between device ID and unique ID.
   *
   * @param deviceId the device ID in Eddystone or iBeacon format.
   * @param uniqueId the device's unique ID.
   * @return ResolveId instance
   */
  public static ResolvedId create(String deviceId, String uniqueId) {
    return new ResolvedId(deviceId, uniqueId);
  }

  private ResolvedId(final String deviceId, final String uniqueId) {
    this.deviceId = deviceId;
    this.uniqueId = uniqueId;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof ResolvedId)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    ResolvedId resolvedId = (ResolvedId) o;
    return SDKEqualsBuilder.start().equals(deviceId, resolvedId.deviceId).equals(uniqueId, resolvedId.uniqueId).result();
  }

  @Override
  public int hashCode() {
    if (hashCode == null) {
      hashCode = HashCodeBuilder.init().append(deviceId).append(uniqueId).build();
    }
    return hashCode;
  }

  /**
   * Returns device ID in iBeacon format.
   *
   * @return the iBeacon-ID instance.
   * @throws IllegalArgumentException if device ID has format other than iBeacon.
   */
  public IBeaconId getIBeaconId() {
    String[] splittedBeaconId = deviceId.split(ID_SEPARATOR);
    SDKPreconditions.checkArgument(splittedBeaconId.length == 3, "Incorrect iBeacon ID format!");

    UUID proximity = UUID.fromString(splittedBeaconId[0]);
    int major = Integer.parseInt(splittedBeaconId[1]);
    int minor = Integer.parseInt(splittedBeaconId[2]);

    return IBeaconId.of(proximity, major, minor);
  }

  /**
   * Returns device ID in Eddystone format.
   *
   * @return the Eddystone-UID instance.
   * @throws IllegalArgumentException if device ID has format other than Eddystone.
   */
  public EddystoneUid getEddystoneUID() {
    String[] splittedEddystoneUID = deviceId.split(ID_SEPARATOR);
    SDKPreconditions.checkArgument(splittedEddystoneUID.length == 2, "Incorrect eddystone UID format!");

    String namespace = splittedEddystoneUID[0];
    String instanceId = splittedEddystoneUID[1];

    return EddystoneUid.of(namespace, instanceId);
  }

  public String getUniqueId() {
    return uniqueId;
  }
}
