package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.google.gson.annotations.SerializedName;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.UUID;

/**
 * This class represents secure proximity mapping. It consists of proximity UUID with correlative
 * secure proximity UUID.
 * <br><br>
 * We can obtain secure proximities mapping via {@link IKontaktCloud}.
 * <br><br>
 * To create new instance of this class, please use {@link ProximityId.Builder}.
 */
public class ProximityId implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<ProximityId> CREATOR = new Creator<ProximityId>() {
    @Override
    public ProximityId createFromParcel(Parcel source) {
      return new ProximityId(source);
    }

    @Override
    public ProximityId[] newArray(int size) {
      return new ProximityId[size];
    }
  };

  @SerializedName("proximity")
  private UUID proximityUUID;

  @SerializedName("secureProximity")
  private UUID secureProximityUUID;

  private boolean shuffled;
  private boolean shared;

  public static Builder builder() {
    return new Builder();
  }

  private ProximityId() {
    this(new Builder());
  }

  private ProximityId(Builder builder) {
    this.proximityUUID = builder.proximityUUID;
    this.secureProximityUUID = builder.secureProximityUUID;
    this.shuffled = builder.shuffled;
    this.shared = builder.shared;
  }

  protected ProximityId(Parcel in) {
    this.proximityUUID = (UUID) in.readSerializable();
    this.secureProximityUUID = (UUID) in.readSerializable();
    this.shuffled = in.readByte() != 0;
    this.shared = in.readByte() != 0;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof ProximityId)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    ProximityId other = (ProximityId) o;
    return SDKEqualsBuilder.start()
        .equals(proximityUUID, other.proximityUUID)
        .equals(secureProximityUUID, other.secureProximityUUID)
        .equals(shuffled, other.shuffled)
        .equals(shared, other.shared)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(proximityUUID).append(secureProximityUUID).append(shuffled).append(shared).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.proximityUUID);
    dest.writeSerializable(this.secureProximityUUID);
    dest.writeByte(shuffled ? (byte) 1 : (byte) 0);
    dest.writeByte(shared ? (byte) 1 : (byte) 0);
  }

  public UUID getProximityUUID() {
    return proximityUUID;
  }

  public UUID getSecureProximityUUID() {
    if (proximityUUID.equals(secureProximityUUID)) {
      return null;
    }
    return secureProximityUUID;
  }

  public boolean isShuffled() {
    return shuffled;
  }

  public boolean isShared() {
    return shared;
  }

  public static class Builder {
    private UUID proximityUUID;
    private UUID secureProximityUUID;
    private boolean shuffled;
    private boolean shared;

    public Builder proximity(final UUID proximityUUID) {
      this.proximityUUID = proximityUUID;
      return this;
    }

    public Builder secureProximity(final UUID secureProximityUUID) {
      this.secureProximityUUID = secureProximityUUID;
      return this;
    }

    public Builder shuffled(final boolean shuffled) {
      this.shuffled = shuffled;
      return this;
    }

    public Builder shared(final boolean shared) {
      this.shared = shared;
      return this;
    }

    public ProximityId build() {
      return new ProximityId(this);
    }
  }
}
