package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.google.gson.annotations.SerializedName;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

/**
 * This class represents secure namespace mapping. It consists of namespace ID with correlative
 * secure namespace ID.
 * <br><br>
 * We can obtain secure namespaces mapping via {@link IKontaktCloud}.
 * <br><br>
 * To create new instance of this class, please use {@link Namespace.Builder}.
 */
public class Namespace implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Namespace> CREATOR = new Creator<Namespace>() {
    @Override public Namespace createFromParcel(Parcel source) {
      return new Namespace(source);
    }

    @Override public Namespace[] newArray(int size) {
      return new Namespace[size];
    }
  };

  @SerializedName("namespace")
  private String namespaceId;

  @SerializedName("secureNamespace")
  private String secureNamespaceId;

  private boolean shuffled;
  private boolean shared;

  private Namespace() {
    this(new Builder());
  }
  public static Builder builder() {
    return new Builder();
  }

  private Namespace(Builder builder) {
    this.namespaceId = builder.namespaceId;
    this.secureNamespaceId = builder.secureNamespaceId;
    this.shuffled = builder.shuffled;
    this.shared = builder.shared;
  }

  protected Namespace(Parcel in) {
    this.namespaceId = in.readString();
    this.secureNamespaceId = in.readString();
    this.shuffled = in.readByte() != 0;
    this.shared = in.readByte() != 0;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof Namespace)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    Namespace other = (Namespace) o;
    return SDKEqualsBuilder.start()
        .equals(namespaceId, other.namespaceId)
        .equals(secureNamespaceId, other.secureNamespaceId)
        .equals(shuffled, other.shuffled)
        .equals(shared, other.shared)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(namespaceId)
        .append(secureNamespaceId)
        .append(shuffled)
        .append(shared)
        .build();
  }

  @Override public int describeContents() {
    return 0;
  }

  @Override public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.namespaceId);
    dest.writeString(this.secureNamespaceId);
    dest.writeByte(shuffled ? (byte) 1 : (byte) 0);
    dest.writeByte(shared ? (byte) 1 : (byte) 0);
  }

  public String getNamespaceId() {
    return namespaceId;
  }

  public String getSecureNamespaceId() {
    return secureNamespaceId;
  }

  public boolean isShuffled() {
    return shuffled;
  }

  public boolean isShared() {
    return shared;
  }

  public static class Builder {
    private String namespaceId;
    private String secureNamespaceId;
    private boolean shuffled;
    private boolean shared;

    public Builder namespace(final String namespaceId) {
      this.namespaceId = namespaceId;
      return this;
    }

    public Builder secureNamespace(final String secureNamespaceId) {
      this.secureNamespaceId = secureNamespaceId;
      return this;
    }

    public Builder shuffled(final boolean shuffled) {
      this.shuffled = shuffled;
      return this;
    }

    public Builder shared(final boolean shared) {
      this.shared = shared;
      return this;
    }

    public Namespace build() {
      return new Namespace(this);
    }
  }
}
