package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SDKPreconditions;

/**
 * An abstraction for concrete events that can be collected.
 * <br><br>
 * We can collect events via {@link IKontaktCloud}.
 */
public abstract class Event implements Parcelable {

  protected final EventType eventType;
  protected final long timestamp;
  protected final String uniqueId;

  protected Event(EventType eventType, long timestamp, String uniqueId) {
    this.eventType = eventType;
    this.timestamp = timestamp / 1000;
    this.uniqueId = uniqueId;
  }

  protected Event(Parcel in) {
    int tmpEventType = in.readInt();
    this.eventType = tmpEventType == -1 ? null : EventType.values()[tmpEventType];
    this.timestamp = in.readLong();
    this.uniqueId = in.readString();
  }

  /**
   * Creates concrete event object based on device and event type.
   * @param device the bluetooth device.
   * @param eventType the event type.
   * @return the event instance.
   */
  public static Event of(final RemoteBluetoothDevice device, final EventType eventType) {
    SDKPreconditions.checkNotNull(device, "device cannot be null");
    SDKPreconditions.checkNotNull(eventType, "event type cannot be null");

    switch (eventType) {
      case SCAN:
        return new ScanEvent(eventType, device.getTimestamp(), device.getUniqueId(), device.getRssi());
      case MONITORING:
        return new MonitoringEvent(eventType, device.getTimestamp(), device.getUniqueId(), device.getBatteryPower());
      default:
        throw new IllegalArgumentException("invalid event type");
    }
  }

  @Override public void writeToParcel(Parcel dest, int flags) {
    dest.writeInt(this.eventType == null ? -1 : this.eventType.ordinal());
    dest.writeLong(this.timestamp);
    dest.writeString(this.uniqueId);
  }

  public EventType getType() {
    return eventType;
  }

  public long getTimestamp() {
    return timestamp;
  }

  public String getUniqueId() {
    return uniqueId;
  }
}
