package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

/**
 * Model class that contains password and master password for {@link Device}.
 * <br><br>
 * We can obtain credentials via {@link IKontaktCloud}.
 */
public class Credentials implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Credentials> CREATOR =
      new Creator<Credentials>() {
        @Override public Credentials createFromParcel(Parcel source) {
          return new Credentials(source);
        }

        @Override public Credentials[] newArray(int size) {
          return new Credentials[size];
        }
      };

  private final String masterPassword;
  private final String password;
  private final String uniqueId;

  /**
   * Constructs new instance of a device's credentials initialized with unique ID, password and master password.
   * @param masterPassword the device's master password.
   * @param password the device's password.
   * @param uniqueId the device's unique ID.
   * @return Credentials instance
   */
  public static Credentials create(String masterPassword, String password, String uniqueId) {
    return new Credentials(masterPassword, password, uniqueId);
  }

  private Credentials(final String masterPassword, final String password, final String uniqueId) {
    this.masterPassword = masterPassword;
    this.password = password;
    this.uniqueId = uniqueId;
  }

  protected Credentials(Parcel in) {
    this.masterPassword = in.readString();
    this.password = in.readString();
    this.uniqueId = in.readString();
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null || !(obj instanceof Credentials)) {
      return false;
    }

    final Credentials credentials = (Credentials) obj;
    return SDKEqualsBuilder.start()
        .equals(uniqueId, credentials.uniqueId)
        .equals(password, credentials.password)
        .equals(masterPassword, credentials.masterPassword)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init()
        .append(uniqueId)
        .append(password)
        .append(masterPassword)
        .build();
  }

  @Override public int describeContents() {
    return 0;
  }

  @Override public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.masterPassword);
    dest.writeString(this.password);
    dest.writeString(this.uniqueId);
  }

  public String getMasterPassword() {
    return masterPassword;
  }

  public String getPassword() {
    return password;
  }

  public String getUniqueId() {
    return uniqueId;
  }

}
