package com.kontakt.sdk.android.cloud.api.executor.firmwares;

import com.kontakt.sdk.android.cloud.api.FirmwaresApi;
import com.kontakt.sdk.android.cloud.api.executor.FileRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.FirmwaresService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import okhttp3.ResponseBody;
import retrofit2.Call;

/**
 * Request executor provided by {@link FirmwaresApi}. Use this class if you want to fetch
 * firmware files through fluent API in chained fashion, for instance:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   String name = "3.1";
 *   FileData firmwareFile = kontaktCloud.firmwares().fileOf(name)
 *      .forDeviceType(DeviceType.BEACON)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that device type must be specified so invocation of {@code forDeviceType} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class FirmwareFileRequestExecutor extends FileRequestExecutor {

  private final FirmwaresService firmwaresService;

  private final String name;
  private DeviceType deviceType;

  /**
   * Constructs request executor initialized with corresponding service class and firmware name.
   * @param firmwaresService the firmwares API service.
   * @param name the firmware name.
   */
  public FirmwareFileRequestExecutor(final FirmwaresService firmwaresService, final String name) {
    this.firmwaresService = firmwaresService;
    this.name = name;
  }

  /**
   * Specifies device type. The method invocation is obligatory while using this request executor.
   * @param deviceType the device type.
   * @return this request executor.
   */
  public FirmwareFileRequestExecutor forDeviceType(final DeviceType deviceType) {
    SDKPreconditions.checkNotNull(deviceType, "device type cannot be null");
    this.deviceType = deviceType;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public FileData execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override public void execute(CloudCallback<FileData> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(deviceType != null, "cannot get firmware - specify device type");
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<ResponseBody> prepareCall() {
    return firmwaresService.getFirmwareFile(name, deviceType.name());
  }
}
