package com.kontakt.sdk.android.cloud.api.executor.events;

import com.google.gson.Gson;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.EventsApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.EventsService;
import com.kontakt.sdk.android.common.model.EventPacket;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link EventsApi}. Use this class if you want to send
 * event packets, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   // define event packet
 *   kontaktCloud.events().collect(eventPacket).execute();
 *   </code>
 * </pre>
 */
public class CollectEventsRequestExecutor extends RequestExecutor<Void> {

  private final EventsService eventsService;

  private final EventPacket eventPacket;

  private final Gson gson = new Gson();

  /**
   * Constructs request executor initialized with corresponding service class and the event packet.
   * @param eventsService the events API service.
   * @param eventPacket the event packet.
   */
  public CollectEventsRequestExecutor(final EventsService eventsService, final EventPacket eventPacket) {
    this.eventsService = eventsService;
    this.eventPacket = eventPacket;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<Void> prepareCall() {
    return eventsService.collectEvents(params());
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Events.TIMESTAMP_PARAMETER, String.valueOf(eventPacket.getTimestamp()));
    params.put(CloudConstants.Events.EVENTS_PARAMETER, gson.toJson(eventPacket.getEvents()));
    return params;
  }
}
