package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to unshare
 * devices through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   kontaktCloud.devices().unshare(IDs)
 *      .fromManagers(managerEmails)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that managers must be specified so invocation of {@code fromManagers} method is
 * mandatory. Otherwise an exception will be thrown at runtime.
 */
public class UnshareDeviceRequestExecutor extends RequestExecutor<String> {

  private final DevicesService devicesService;

  private final String[] uniqueIds;
  private String[] managerEmails;

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public UnshareDeviceRequestExecutor(final DevicesService devicesService, final String... uniqueIds) {
    this.devicesService = devicesService;
    this.uniqueIds = uniqueIds;
  }

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public UnshareDeviceRequestExecutor(final DevicesService devicesService, final List<String> uniqueIds) {
    this.devicesService = devicesService;
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
  }

  /**
   * Specifies managers. The method invocation is obligatory while using this request executor.
   * @param managerEmails manager email addresses.
   * @return this request executor.
   */
  public UnshareDeviceRequestExecutor fromManagers(final String... managerEmails) {
    SDKPreconditions.checkNotNull(managerEmails, "manager emails cannot be null");
    this.managerEmails = managerEmails;
    return this;
  }

  /**
   * Specifies managers. The method invocation is obligatory while using this request executor.
   * @param managerEmails manager email addresses.
   * @return this request executor.
   */
  public UnshareDeviceRequestExecutor fromManagers(final List<String> managerEmails) {
    SDKPreconditions.checkNotNull(managerEmails, "manager emails cannot be null");
    final int size = managerEmails.size();
    this.managerEmails = managerEmails.toArray(new String[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<String> prepareCall() {
    return devicesService.unshareDevices(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(managerEmails != null, "cannot share device - specify managers");
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    params.put("managerMail", StringUtils.join(managerEmails, ","));
    return params;
  }
}
