package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.response.paginated.Devices;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Access;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.Map;
import java.util.UUID;

abstract class DevicesBaseRequestExecutor extends PaginatedRequestExecutor<Devices> {

  protected Access access;
  protected DeviceProfile profile;
  protected DeviceType type;
  protected UUID[] managerIds;

  /**
   * Specifies access parameter.
   * @param access the access.
   * @return this request executor.
   */
  public DevicesBaseRequestExecutor access(final Access access) {
    this.access = SDKPreconditions.checkNotNull(access, "access cannot be null");
    return this;
  }

  /**
   * Specifies managers for whom devices are assigned to.
   * @param managerIds manager unique identifiers.
   * @return this request executor.
   */
  public DevicesBaseRequestExecutor managerIds(final UUID... managerIds) {
    this.managerIds = SDKPreconditions.checkNotNull(managerIds, "manager IDs cannot be null");
    return this;
  }

  /**
   * Specifies managers for whom devices are assigned to.
   * @param managerIds manager unique identifiers.
   * @return this request executor.
   */
  public DevicesBaseRequestExecutor managerIds(final List<UUID> managerIds) {
    SDKPreconditions.checkNotNull(managerIds, "manager IDs cannot be null");
    final int size = managerIds.size();
    this.managerIds = managerIds.toArray(new UUID[size]);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (access != null) {
      params.put(CloudConstants.Common.ACCESS_PARAMETER, access.name());
    }
    if (profile != null) {
      params.put(CloudConstants.Devices.PROFILE_PARAMETER, profile.name());
    }
    if (type != null) {
      params.put(CloudConstants.Devices.DEVICE_TYPE_PARAMETER, type.name());
    }
    if (managerIds != null) {
      params.put(CloudConstants.Devices.MANAGER_ID_PARAMETER, StringUtils.join(managerIds, ","));
    }
    return params;
  }
}
