package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.response.paginated.CredentialsList;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to get
 * device credentials list through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   CredentialsList credentialsList = kontaktCloud.devices().credentials(deviceIDs).execute();
 *   </code>
 * </pre>
 */
public class CredentialsListRequestExecutor extends RequestExecutor<CredentialsList> {

  private final DevicesService devicesService;

  private final List<String> uniqueIds;

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public CredentialsListRequestExecutor(final DevicesService devicesService, final List<String> uniqueIds) {
    this.devicesService = devicesService;
    this.uniqueIds = uniqueIds;
  }

  /**
   * {@inheritDoc}
   */
  @Override protected Call<CredentialsList> prepareCall() {
    return devicesService.getDeviceCredentialsByUniqueIds(StringUtils.join(uniqueIds, ","));
  }

  @Override protected Map<String, String> params() {
    return new HashMap<>();
  }
}
